%Pkg = (name    => "BIC-Summary",
        version => "2.1");
        # December 2012

%Write = ("html"        => 1,
          "summary"     => 1,
          "xls"         => 1);


%PaperFmt = ("main"    => "legal",
             "summary" => "legal");
             # each of these variables can take one of the values "legal" or "letter",
             # depending on the type of paper you may eventually want to write the resulting
             # documents (this pgm produces both a quite complete bic output [main]
             # and a summary, useful when an outcome variable was modeled in a relatively large number
             # of ways [summary], e.g., for different patients subsets)


%xlsfmts = (
            0             => {align => right},
            analysisno    => {align => center,
                              bold  => 1,
                              color => orange},
            blocklbl      => {bold  => 1},
            catlbl        => {align => center},
            details       => {align => left,
                              color => 44},
            firstmodelin  => {align => center},
            firstmodelout => {align => center, color => 53},
            forcedin      => {align  => center,
                              color  => 38,
                              italic => 1},
            indepvarname  => {align    => center,
                              bg_color => 38,
                              bold     => 1,
                              color    => white},
            lcl           => {align  => right,
                              color  => gray,
                              italic => 1},
            logoutcomename => {color => 17,
                               bold  => 1},
            modelno       => {align => center},
            notes         => {align  => left,
                              color  => 18,
                              italic => 1},
            outcomename   => {align    => left,
                              bg_color => 17,
                              bold     => 1,
                              color    => white},
            postmoments   => {align => center,
                              color => 19},
            priorprob     => {align => center,
                              color => 50},
            probne0       => {align => center},
            refcatlbl     => {align => center,
                              color => 44},
            rejected      => {align    => center,
                              color    => 37,
                              fg_color => silver,
                              font     => 'Lucida Handwriting',
                              pattern  => 14,
                              valign   => vcenter},
            rejectedpriorprob => {align => center,
                                  color => 50},
            rejectedvarname  => {align    => center,
                                 bg_color => 38,
                                 bold     => 1,
                                 color    => white},
            repeatedfrom => {color => 17},
            summarycomment => {align => right,
                               color => silver},
            summaryindep => {align => right,
                             color => 17},
            summarylbl => {align => left,
                           italic => 1},
            summarylblindep => {align => left,
                                bold  => 1,
                                color => 17,
                                italic => 1},
            summarynolbl => {align  => center,
                             color  => orange,
                             italic => 1},
            toplbl        => {align => right},
            ucl           => {align  => left,
                              color  => gray,
                              italic => 1},
            writtenby     => {color  => 19,
                              italic => 1},


            nilcatlbl       => {align => center},
            nilfirstmodelin => {align => center},
            nilfirstmodelout => {align => center, color => 53},
            nilindepvarname => {align    => center,
                                bg_color => 38,
                                bold     => 1,
                                color    => white},
            nilpostmoments   => {align => center,
                                 color => 19},
            nilpriorprob => {align => center,
                             color => 50},
            nilprobne0 => {align => center},
            nilrefcatlbl => {align => center,
                             color => 44},

            logyvar => {color => 17, bold => 1},
            logrepfrom => {color => 17},
            logchange => {color => 20, italic => 1}
           );


%colspan = (
            blocklbl      => 3,
            catlbl        => 3,
            firstmodelin  => 3,
            firstmodelout => 3,
            forcedin      => 3,
            indepvarname  => 3,
            postmoments   => 3,
            priorprob     => 3,
            probne0       => 3,
            refcatlbl     => 3
            );

@KeepValuesAsIsForThese = ("analysisno", "blocklbl", "catlbl", "forcedin",
                           "indepvarname",
                           "logchange", "logrepfrom", "logyvar",
                           "refcatlbl", "rejected",
                           "summaryindep");


%xlsInfo = (noScientificationNotationWhenELowerOrEqualTo => 4);


%Color = (
          NA            => "999999", # text color for 'rejected'/'n/a' for rejected/not candidate variables
          neutral       => CCCCCC,   # used to separate values in categorical variables
          notes         => "000066", # color used for notes (ifs and other notes that could have been added through notes= option in modified.bicout
          othervars     => "000099", # color for border for Other variables section
          othervarsbg   => CCCCFF,   # background color for Other variables section
          postmomentsbg => FFFFFF,   # background color in posterior moments section
          "ref"         => F08080,   # color used for reference labels
          sumanalysisno => FF9900,   # background color for column identifying output no
          sumtitle      => "000066", # main color
          title         => "006400", # main color (variables list section, on top of each table of parameter estimates)
          underborder   => "000066"  # border color separating rejecting vars and other vars from main table
          );

@COLORS = ("FFF8DC", "E6DFC5"); # alternating colors for lines in parameters' estimates tables

# probne0's summary tables colors

@SUMCOLORS = ("FFFFFF", "FFCC99"); # alternating lines' colors

$MODELPROBNDIGITS      =  2;
$PARMSESTIMATESNDIGITS =  3;

$MAXNMODELS2PRESENT    = 20;

$AVOIDEMPTYTABLES = 1;       # For html output files only:
                             # When $MAXNMODELS2PRESENT is (relatively small), some parts of the MLE's table may
                             # be empty and use some space to report... nothing!
                             # Set to 1 to avoid such empty space (and report postmean, postsd and prior prob for these variables
                             # in a mini-table, on the side, to save space)

$OTHERVARSALIGN   = "right"; # where to place the subtable Other variables relatively to the main parameters table
                             # (could be left, middle, right)


$NBESTMODELSPROBS = 3;       # number of best models probs to report (in probne0 summary table)

$TRUNCEXT         = 1;       # whether or not extension of outcome names should be truncated in html output
                             # >> comment out if your bic output names do not end with "somesuffix.outputnumber"
                             # (I personally sometimes name similar bic analyses for a same outcome variable [myout, say]
                             # myout.res.1, myout.res.2, myout.res.3, etc, and by setting $RM_RES_EXTENSIONS to 1 here,
                             # the html summary will not report the full names myout.res.1, myout.res.2 and myout.res.3
                             # but rather myout [1], myout [2] and myout [3], which is easier to understand for the reader)
                             # -> Extension will be removed only if it is in %IsRemovableExtension

# a few font sizes...


$FONTSIZE          = "11px";
$TITLEFONTSIZE     = "16px";
$OTHERVARSFONTSIZE = "12px";

$AND = " <b>\&amp;</b> ";     # 'text' to be displayed between the two categories labels in interaction terms
                              # (involving two categorical variables)

@RemovableExtensions = ("res");


# ------ Do not edit below this line ----------------------------------------------------------------------------


%xls = (
        "afterheaderblanknrows" => 1,
        "leftblockncolumns"     => 3,
        "sepblankrows"          => 2   # number of blank lines bewteen two different BIC runs (for a same outcome variable)
       );


$CENSORINGVAR          = "censoring.var";
$FAMILY                = "glm.family";
$FIRSTMODELIN          = "first.model.in";
$FIRSTMODELOUT         = "first.model.out";
$FORCEDIN              = "forced.in";
$HAZARD                = "hazard.ratio";
$HEAD                  = "model.head";
$MLE                   = "mle";
$MODEL                 = "label";
$MODELPROB             = "postprob";
$NEVENTS               = "n.events";
$NOTES                 = "notes";
$OR                    = "odds.ratio";
$POSTMEAN              = "postmean";
$POSTSD                = "postsd";
$PRIORPROB             = "prior.param";
$PROBNE0               = "probne0";
$RATERATIO             = "rate.ratio";
$REF                   = "reference.values";
$REJECTEDVARS          = "rejected.vars";
$REJECTEDVARSPRIORPROB = "rejected.vars.prior.param";
$SAMPLESIZE            = "n";
$VARNAMES              = "vars.in.model";

%SHEET = (log => "_Log", summary => "_Summary");

$DimSynonym{"family"} = $FAMILY;

# IMPORTANT
# ---------

# Define which of constants defined in block above have model-level corresponding hashes
# (that is, where 4th index of %info refers to model#)

foreach ($MODEL, $MODELPROB)
{
  $IsModelLevel{$_} = 1;
}

# Also which of dimensions defined in block above refer to character string infos (will be quoted text in bic output)

foreach ($NOTES)
{
  $ReadStrings{$_} = 1;
}

$LABELSEQ  = "=";  # was "::" in earlier versions
$LABELSDIV = "::";


# -------------------------------------------------------------------------------------------------------------------


%Section = ($MLE, "Parameter estimates", $OR, "Odds ratios", $HAZARD, "Hazard ratios", $RATERATIO, "Rate ratios");

%Style = ("a.insight",
            "font-style: italic; color: blueviolet;",

          "span.insight",
            "font-style: italic; color: blueviolet;",

          "table.bic",
            "font-size=$FONTSIZE;",

          "tr.postmoments",
            "text-align: right; background-color: $Color{postmomentsbg}; padding-top: 2px;",
          "tr.priorprob",
            "text-align: right; background-color: $Color{postmomentsbg};",
          "tr.probne0title",
            "background-color: $Color{sumtitle}; color: white; text-align:left; padding: 0 4px 0 4px;",
          "tr.probne0subtitle",
            "border-style: solid; border-color: $Color{sumtitle}; border-width: 2px 0 1px 0;",
          "tr.rejectedvars",
            "text-align: left; background-color: $Color{postmomentsbg};",
          "tr.section",
            "font-size: $TITLEFONTSIZE; font-weight: bold; padding-top: 4px;",

          "td.emptytitle",
            "border-style: solid; border-color:$Color{title}; border-width: 2px 0 0 0; font-style: italic; color: $Color{title}; text-align: left;",
          "td.family",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: left; color: $Color{title}; font-size: $FONTSIZE;",
          "td.firstin",
            "padding-left: 4px; border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: left;",
          "td.firstin4cat",
            "padding-left: 4px; border-style: solid; border-color:$Color{neutral}; border-width: 0 1px 0 0; text-align: left;",
          "td.forcedin",
            "text-align: left; color: $Color{title}; font-style:italic; border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0;",
          "td.forcedin4cat",
            "text-align: left; color: $Color{title}; font-style:italic; border-style: solid; border-color:$Color{neutral}; border-width: 0 1px 0 0;",
          "td.infocell0",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: right; font-style: italic;",
          "td.infocell0postm",
            "border-style: solid; border-color:$Color{title}; border-width: 1px 1px 0 0; text-align: right; font-style: italic;",
          "td.insight",
            "border-style: solid; border-color: slategray; border-width: 1px 1px 1px 1px; background-color: whitesmoke; color: lightslategray; padding: 5px 20px 10px 5px;",
          "td.leftcell",
            "text-align: left; padding-left: 4px;",
          "td.modellabel",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: right; padding-right: 2px;",
          "td.modelno",
            "text-align: right; padding-right: 2px;",
          "td.modelno4cont",
            "text-align: right; padding-right: 2px; border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0;",
          "td.n",
            "text-align: left; color: $Color{title}; font-style: italic;",
          "td.notes",
            "border-style: solid; border-color:$Color{title}; border-width: 1px 1px 2px 4px; color: $Color{notes}; padding: 0 10px 0 4px;",
          "td.notesandlabels",
            "border-style: solid; border-color:$Color{title}; border-width: 1px 1px 2px 4px; color: $Color{notes}; padding-left: 4px;",

          "td.othervarsbottomentry",
            "border-style: solid; border-color: $Color{othervars}; border-width: 0 1px 1px 0; padding: 0 2px 0 2px; text-align: right; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarscatentry",
            "border-style: solid; border-color: $Color{othervars}; border-width: 0 1px 0 0; padding: 0 2px 0 2px; text-align: right; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarsentry",
            "border-style: solid; border-color: $Color{othervars}; border-width: 0 1px 0 0; padding: 0 2px 0 2px; text-align: right; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarsheader0",
            "border-style: solid; border-color: $Color{othervars}; border-width: 1px 1px 0 1px; color: $Color{othervars}; padding: 0 2px 0 2px; text-align: left; font-size: $OTHERVARSFONTSIZE; font-weight: bold; background-color: $Color{othervarsbg};",
          "td.othervarscatheader",
            "border-style: solid; border-color: $Color{othervars}; border-width: 1px 0 0 0; color: $Color{othervars}; padding: 0 2px 0 2px; text-align: left; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarsheader",
            "border-style: solid; border-color: $Color{othervars}; border-width: 1px 1px 0 0; color: $Color{othervars}; padding: 0 2px 0 2px; text-align: left; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarsleft",
            "border-style: solid; border-color: $Color{othervars}; border-width: 0 1px 0 1px; padding: 0 2px 0 2px; text-align: left; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",
          "td.othervarsleftlast",
            "border-style: solid; border-color: $Color{othervars}; border-width: 0 1px 1px 1px; padding: 0 2px 0 2px; text-align: left; font-size: $OTHERVARSFONTSIZE; background-color: $Color{othervarsbg};",

          "td.pointestimate",
            "padding: 0 5px 0 5px; text-align: left;",
          "td.pointestimatepostm",
            "padding: 0 5px 0 5px; text-align: left; border-style: solid; border-color: $Color{title}; border-width: 1px 0 0 0;",
          "td.post",
            "text-align: right;",
          "td.postm",
            "border-style: solid; border-color: $Color{title}; border-width: 1px 0 0 0;",
          "td.postrcell",
            "border-style: solid; border-color: $Color{title}; border-width: 0 1px 0 0; text-align: right; padding-right: 2px;",
          "td.postrcell0",
            "border-style: solid; border-color: $Color{title}; border-width: 1px 1px 0 0; text-align: right; padding-right: 2px;",
          "td.probne0",
            "border-style: solid; border-width: 0 1px 0 0; border-color:$Color{neutral}; font-size: $FONTSIZE; padding-left: 4px;",
          "td.probne0analysisno",
            "background-color: $Color{sumanalysisno}; font-size: $FONTSIZE; padding: 0 4px 0 4px; text-align: right;",
          "td.probne0bestmodel",
            "font-size: $FONTSIZE;",
          "td.probne0bestmodelempty",
            "font-size: $FONTSIZE; background-color: white;",
          "td.probne0indepvarsheader",
            "font: bold italic normal $TITLEFONTSIZE serif;",
          "td.probne0outcomevarheader",
            "font-size: $TITLEFONTSIZE; font-weight: bold;",
          "td.probne0rejected",
            "border-style: solid; border-color: $Color{neutral}; border-width: 0 1px 0 0; text-align: right; padding: 0 2px 0 2px; color: $Color{NA};  font-size: $FONTSIZE;",
          "td.probne0subtitle",
            "font-size: $FONTSIZE; font-style: italic; color: $Color{sumtitle}; border-style: solid; border-color: $Color{sumtitle}; border-width: 2px 0 1px 0;",
          "td.probne0subtitleanalysisno",
            "background-color: $Color{sumanalysisno}; font-size: $FONTSIZE; font-style: italic; padding: 0 4px 0 4px; border-style: solid; border-color: $Color{sumtitle}; border-width: 2px 0 0 0;",
          "td.probne0value",
            "border-style: solid; border-color: $Color{neutral}; border-width: 0 1px 0 0; text-align: right; padding: 0 2px 0 2px; font-size: $FONTSIZE;",
          "td.probne0varname",
            "border-style: solid; border-width: 2px 0 1px 0; border-color:$Color{sumtitle}; font-size: $FONTSIZE; color: $Color{sumtitle}; padding-left: 4px;",
          "td.refvaluelabel",
            "border-style: solid; border-color: $Color{neutral}; border-width: 0 1px 0 0; text-align: left; padding-left: 2px; color: $Color{ref};",
          "td.refvaluelabel1",
            "border-style: solid; border-color: $Color{title}; border-width: 0 1px 0 0; text-align: left; padding-left: 2px; color: $Color{ref};",
          "td.rejectedvarslist",
            "padding-left: 5px; text-align: left; border-style: solid; border-color: $Color{underborder}; border-width: 1px 0 0 0;",
          "td.sectioncell0",
            "text-align: left; font-weight: bold;",
          "td.subtitlecell0",
            "text-align: left; font-weight: bold;",
          "td.subtitlercell",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: right; padding-right: 2px;",
          "td.title",
            "background-color:$Color{title}; font-size: $TITLEFONTSIZE; color: white; font-weight: bold; padding: 0 2px 0 2px;",
          "td.title a",
            "text-decoration: none; font-size: $TITLEFONTSIZE; color: white; font-weight: bold;",
          "td.uppercl",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: left; padding-right: 2px;",
          "td.valuelabel",
            "border-style: solid; border-color:$Color{neutral}; border-width: 0 1px 0 0; text-align: left; padding-left: 2px; color: $Color{title};",
          "td.valuelabel1",
            "border-style: solid; border-color:$Color{title}; border-width: 0 1px 0 0; text-align: left; padding-left: 2px; color: $Color{title};"
         );



@Dims2Report = ($MLE, $OR, $HAZARD, $RATERATIO);

$LEFTCOLSPAN = 3;
$VARCOLSPAN  = 3;

@LEFTWIDTH =  (360,  150); # 2nd component is for cont'd tables
$VARWIDTH  =         120 ;
%MAXWIDTH  =  ("letter", 912, "legal", 1200); # letter and legal max widths

@ModelNoClass = ("modelno", "modelno4cont"); # for original and cont'd parts
@TitleClass   = ("title", "emptytitle");

%MAXNVARSPERBLOCK = ("letter", 5, "legal", 7); # number of columns on letter/legal sheets

foreach (keys %PaperFmt)
{
  $PaperFmt{$_} = lc $PaperFmt{$_};
}

$BLANK = "&nbsp;";

# probne0's summary tables widths

$PROBNE0WIDTH        =  50;
$ANALYSISNOWIDTH     =  40;
$BESTMODELPROBSWIDTH = 120;
$BESTMODELWIDTH      = 300;


# -----------------------------------------------------------------------------------------------------------------------

use Win32;

&OpenSTDERR($0, 1);


while (@ARGV)
{
  $_ = shift;

  if (-f)
  {
    die "Give only one input file name at a time. Abort.\n" if $In;
    $In = $_;
  }
}

die "No input file specified. Abort.\n" unless $In;


($dir, $Out) = Win32::GetFullPathName($In);
$Out = $1 if $Out =~ /(.*)\./;


%tmpOut = ("html"        => "${dir}BIC-$Out.html",
           "htmlsummary" => "${dir}BIC-$Out-summary.html",
           "xls"         => "${dir}BIC-$Out.xls");

($Out{htmlsummaryfname}) = reverse Win32::GetFullPathName($Out{htmlsummary});

$RSrc = "$dir$Out.R";


# ---

foreach $ext (@RemovableExtensions)
{
  $ext = lc $ext;
  $IsRemovableExtension{$ext} = 1;
}

# ---



open(ROUT, $In) || die "Cannot read R Output file ($In). Abort.\n";

# Read preamble, if present (preamble, optional, would be a list of values copied in R output that user wanted to pass to bic2html, -------------------------
# e.g., maxnmodels2present

while (<ROUT>)
{
  last if /Left with/;
  chomp;
  push(@preamble, $_);
}


if (/Left with/)
{
  while (@preamble)
  {
    $_ = shift @preamble;

    if (/\s*\$maxnmodels2present/ && !$`)
    {
      $_ = shift @preamble;
      $MAXNMODELS2PRESENT = $1 if /\]\s+(\d+)/;
    }
  }

  @rout = (<ROUT>);
}
else
{
  @rout = @preamble;
}

close ROUT;

# ------------------------------------------------------------------------------
&ReadInsightsFromRSrc();


# ------------------------------------------------------------------------------


while (@rout)
{
  $_ = shift @rout;

  chomp;
  /\S+/;
  $field1 = $&;

  if (/\s*\>\s*(\S+)\s*/ && !$` && !$')
  {
    $outcome = $1;
  }
  elsif (/\s*\$(\S+)\s*/ && !$` && !$')
  {
    $dim = $1;
    $dim = $DimSynonym{$dim} if $DimSynonym{$dim};
  }
  elsif (/\[\s*(\d+)\s*(,?)\s*\]/)
  {
    $modelno = $2 eq "," ? $1 : 0;
    $nmodels{$outcome} = $modelno if $modelno > $nmodels{$outcome};

    next if $MAXNMODELS2PRESENT && ($modelno > $MAXNMODELS2PRESENT);
    next if $MAXNMODELS2PRESENT && $IsModelLevel{$dim} && $k{$outcome}{$dim}{0} >= $MAXNMODELS2PRESENT;

    &ReadColumnLabels($outcome, $dim, $_, $previousline, $k{$outcome}{$dim}{1}) if $modelno == 1;

    $_ = $';
    s/\s*(.*)/\1/g; # ltrim
    s/\s+/ /g;


    if ($dim eq $HEAD)
    {
      s/[^"]*"//;
      s/".*//;
      s/\s+//g;
    }


    if ($ReadStrings{$dim})
    {
      s/"\s*"/\t/g;
      s/"//g;
    }
    else
    {
      s/\s/\t/g;
    }


    @tmp = split(/\t/, $_);

    foreach (@tmp)
    {
      s/"//g;
      s/,/, /g if $dim eq $MODEL;
      $info{$outcome}{$dim}{$modelno}{++$k{$outcome}{$dim}{$modelno}} = $_;
      $AnyInfo{$outcome}{$dim} = 1;
    }
  }
  elsif ($field1 eq "NA" || (/\s*["\-\d]/ && !$`))
  {
    next if $IsModelLevel{$dim} && $k{$outcome}{$dim}{$modelno} > $MAXNMODELS2PRESENT;

    $modelno = 0;

    if (/"/)
    {
      while (/"/)
      {
        $_ = $';
        /"/;
        ($info{$outcome}{$dim}{$modelno}{++$k{$outcome}{$dim}{$modelno}}, $_) = ($`, $');
        $AnyInfo{$outcome}{$dim} = 1;
      }
    }
    else
    {
      @tmp = split;

      foreach (@tmp)
      {
        $info{$outcome}{$dim}{$modelno}{++$k{$outcome}{$dim}{$modelno}} = $_;
        $AnyInfo{$outcome}{$dim} = 1;
      }
    }
  }
  else
  {
    $previousline = $_;
  }
}


&ScanOutcomeNames(keys %info);
@outcomevars = sort {&MixedOrder($Y{$a}{short}, $Y{$b}{short}) || &MixedOrder($Y{$a}{displayed}, $Y{$b}{displayed}) || $Y{$a}{no} <=> $Y{$b}{no}} keys %info;


&WriteHtmlFile()  if $Write{html};
&WriteExcelFile() if $Write{xls};


# --- (the end) -------------------------


print STDERR "Output sent to file(s):\n";
@tmp = sort {lc $a cmp lc $b} keys %Out;
foreach (@tmp)
{
  print STDERR "\t" . $Out{$_} . "\n";
}





die "Done.\n";


# ---- End of code ------------------------------------------------------------------------------------------


sub Family()
{
  my ($outcomevar) = @_;
  my ($tmp);
  
  $tmp = $info{$outcomevar}{$FAMILY}{0}{1};
  unless ($tmp)
  {
    $tmp = $info{$outcomevar}{$CENSORINGVAR}{0}{1};
    unless ($tmp)
    {
      $tmp = $info{$outcomevar}{$HEAD}{0}{1};
      $tmp =~ s/.*,\s*//;
      $tmp =~ s/\s*\)//;
    }

    $tmp = "censoring variable: $tmp";
  }
  
  $tmp;
} # end of Family


sub Max2()
{
  my ($a, $b) = @_;
  
  $a > $b ? $a : $b;
} # end of Max2


sub MixedOrder()
{
  my ($u, $v) = @_;
  my (%alpha, %num);
  
  $u =~ s/^\s+|\s+$//g; # lrtrim
  $v =~ s/^\s+|\s+$//g; # lrtrim
  
  if ($u eq $v)
  {
    0;
  }
  else
  {
    foreach ($u, $v)
    {
      if (/\d+/)
      {
        ($alpha{$_}, $num{$_}, $rest{$_}) = ($`, $&, $');
      }
      else
      {
        $alpha{$_} = $_;
      }
    }
    
    lc $alpha{$u} cmp lc $alpha{$v} || $num{$u} <=> $num{$v} || &MixedOrder($rest{$u}, $rest{$v});
  }
} # end of MixedOrder


sub ModelsShortList()
{
  my ($maxno, @nos) = @_;

  if ($maxno)
  {
    pop @nos while @nos && $nos[$#nos] > $maxno;
  }

  @nos;
} # end of ModelsShortList


sub NotScientific()
{
  my ($x) = @_;
  my ($e, $lx, $negative, $touched, $xroot);
  
  # Global variables
  # -----------------
  # %xlsInfo

  if ($x =~ /-?\d+\.\d+e\-\d+/i || $x =~ /-?\d+e\-\d+/i)
  {
    $x =~ /e/i;
    ($xroot, $e) = ($`, $');

     if ($e >= -1 * $xlsInfo{noScientificationNotationWhenELowerOrEqualTo})
     {
       ($x, $touched) = ($xroot, 1);
       $x =~ s/\.//;
       ($negative, $x) = (1, substr($x,1)) if substr($x,0,1) eq "-";
    
       $x = "0." . "0" x (-$e-1) . $x;
     }
  }
  elsif ($x =~ /-?\d+\.\d+e\+\d+/i || $x =~ /-?\d+e\+\d+/i)
  {
    $x =~ /e/i;
    ($xroot, $e) = ($`, $');
    
    if ($e <= $xlsInfo{noScientificationNotationWhenELowerOrEqualTo})
    {
      ($x, $touched) = ($xroot, 1);
      $x =~ s/\.//;
      ($negative, $x) = (1, substr($x,1)) if substr($x,0,1) eq "-";
      $e++;
    
      $lx = length($x);
      if ($e < $lx)
      {
        substr($x, $e, 0) = ".";
      }
      elsif ($e > $lx)
      {
        $x .= "0" x ($e-$lx);
      }
    }
  }

  $x = "-" . $x if $negative && $touched;
  $x;
} # end of NotScientific


sub NSame()
{
  my ($what, @in) = @_;
  my ($n);

  $n = 0;

  foreach (@in)
  {
    last unless $_ eq $what;
    $n++;
  }

  $n;
} # end of NSame


sub OpenSTDERR()
{
  my ($callingpgm, $timestamp, $definegloballogfile) = @_;
  my ($logfile, $logdir);

  # Global variables defined:
  # ------------------------------------------------------- 
  # $LogFile (optionally, that is, if $definegloballogfile)
  # $MyStartTime


  $logfile = "$1.log" if $callingpgm =~ /(.*)\./;
  ($logdir, $logfile) = Win32::GetFullPathName($logfile);
  $logdir .= "log\\" if lc $logdir eq "c:\\patrick\\bin\\";
  $logfile = $logdir . $logfile;

  open(STDERR, ">$logfile") || die "Cannot write to log file $logfile. Abort.\n";
  print STDERR localtime() . "\n\n" if $timestamp;

  $MyStartTime = time; # defined for future use, as in fct ReportRunTime, for example

  $LogFile = $logfile if $definegloballogfile;
}


sub ReadColumnLabels()
{
  my ($outcome, $dim, $parms, $labels, $lastk) = @_;
  my ($label, $l, $tmp, $tmplabel, $tmpvar, $valuelabel,
      @tmplabels, @tmpvars, @varsvalues);

  $_ = $parms;
  /\]/;
  $_ = $';
  $labels = substr($labels, length($`) + 1);
  $k = $lastk;

  while (/\S+/)
  {
    $_ = $';
    $l = length($` . $&);
    $label  = substr($labels, 0, $l);
    $labels = substr($labels, $l);
    $label =~ s/\s*(.*)/\1/g; # ltrim

    $k++;
    undef $valuelabel;
    undef $var;
    undef @varsvalues;

    if ($label =~ /$LABELSEQ/o)
    {
      ($var, $valuelabel) = ($`, $');

      @tmpvars = split(/\*/, $var);
      @tmplabels = split(/\:\:/, $valuelabel);

      if ($#tmpvars > 0 && $#tmpvars == $#tmplabels)
      {
        while (@tmpvars)
        {
          ($tmpvar, $tmplabel) = (shift @tmpvars, shift @tmplabels);
          push(@varsvalues, "$tmpvar=$tmplabel");
        }

        $valuelabel = join($AND, @varsvalues) if @varsvalues;
      }
    }
    else
    {
      $var = $label;
    }    


    $col0{$outcome}{$dim}{$var} = $k unless $col0{$outcome}{$dim}{$var};
    $col1{$outcome}{$dim}{$var} = $k;

    if ($dim eq $MLE && $valuelabel)
    {
      $valuelabel{$outcome}{$var}{++$ncol{$outcome}{$var}} = $valuelabel;
      $catvarno{$outcome}{$var} = ++$ncatvars{$outcome} if $ncol{$outcome}{$var} == 1;
    }

    $var{$outcome}{$dim}{$k} = $var;
  }
} # end of ReadColumnLabels


sub ReadInsightsFromRSrc()
{
  my (@comments, @rcode,
      $changes, $outname, $repetitionof, $tmp);
      
  # Global hash tables
  # ------------------
  # %changes
  # %IsRepetitionOf
  # %Repeated
  
  if (-e $RSrc)
  {
    open(TMP, $RSrc);
    @rcode = (<TMP>);
    close TMP;
    
    while (@rcode)
    {
      $tmp = shift @rcode;
      last if $tmp =~ /#\s*insights\s*\:/i;
    }
    
    foreach (@rcode)
    {
      last unless /\s*#/ && !$`;
      /#\s*(\S+)\s+(\S+)\s+/;
      ($outname, $repetitionof, $changes) = ($1, $2, $');
      $changes =~ s/\s+$//; # rtrim
      ($IsRepetitionOf{$outname}, $changes{$outname}, $Repeated{$repetitionof}) = ($repetitionof, $changes, 1);
    }
  }
} # end of ReadInsightsFromRSrc


sub RejectedPriorProb()
{
  my ($outcomevar, $rejectedvar) = @_;
  my ($no, $prob);
  
  foreach $no (keys %{$info{$outcomevar}{$REJECTEDVARS}{0}})
  {
    if ($info{$outcomevar}{$REJECTEDVARS}{0}{$no} eq $rejectedvar)
    {
      $prob = $info{$outcomevar}{$REJECTEDVARSPRIORPROB}{0}{$no}; 
      last;
    }
  }

  $prob;
} # end of RejectedPriorProb


sub Round()
{
  my (@z) = @_;
  my ($x, $e);

  foreach (@z)
  {
    next if /inf/i;

    ($x, $e) = ($_, "");
    ($x, $e) = ($`, $& . $') if /e/i;
    
    if (!$e && $x =~ /0\.0/)
    {
      $x = sprintf("%.${PARMSESTIMATESNDIGITS}e", $x);
    }
    else
    {
      $x = sprintf("%.${PARMSESTIMATESNDIGITS}f", $x);
    }

    $_ = $x . $e;
    s/(e[\+\-])0+(\d)/\1\2/;
    s/(e[\+\-])0+//;
  }

  @z;
} # end of Round


sub ScanOutcomeNames()
{
  my (@outcomenames) = @_;
  my ($displayed, $ext, $html, $outcomename, $short);
  
  # Global variables:
  # -----------------
  # %IsRemovableExtension
  # %Y
  
  foreach $outcomename (@outcomenames)
  {
    $short = $outcomename;
  
    ($short, $Y{$outcomename}{no}) = ($`, $1) if $short =~ /\.(\d+)/ && !$';
  
    
    if ($TRUNCEXT && $short =~ /(.*)\./)
    {
      $ext = lc $';
      $short = $1 if $IsRemovableExtension{$ext};
    }
    
    ($short, $Y{$outcomename}{details}) = ($`, $') if $short =~ /[^\w\.]/;

    $Y{$outcomename}{short} = $short;
    
    $displayed = $short;
    $displayed .= " " . $Y{$outcomename}{details} if $Y{$outcomename}{details};
    $html = $displayed;
    
    if (defined $Y{$outcomename}{no})
    {
      $html      .= "&nbsp;&nbsp;&nbsp;\[" . $Y{$outcomename}{no} . "\]";      
      $displayed .= " \[" . $Y{$outcomename}{no} . "\]";
    }
    else
    {
      $Y{$outcomename}{no} = -1; # better than undefined, for future sorting
    }

    ($Y{$outcomename}{html}, $Y{$outcomename}{displayed}) = ($html, $displayed);
  }
} # end of ScanOutcomeNames


sub WhichVarIndex()
{
  my ($varname, %varnames) = @_;
  my ($which);

  $which = -1;
  foreach (keys %varnames)
  {
    if ($varnames{$_} eq $varname)
    {
      $which = $_;
      last;
    }
  }

  $which;
} # end of WhichVarIndex


sub WriteExcelFile()
{
   my ($action, $analysisno, $c, $changes, $col, $dim, $dim0, $dim1, $fmtname, $ind,
       $lcl, $k, $lastmodelno, $modelno, $modelprob, $modulo, $nil, $notes, $outcomevar, 
       $parmcol, $pointestimate, $row, $short, $tmp, $ucl, $val, $var, $width, $xlsrow,
       @changes, @ind, @k, @modelnos, 
       @tmp, @val, @valuenos, @variables, @varindices,
       %inreportedmodels, %isforcedin, %KeepValuesAsIs, %logMsg, %rejected, %tmpforcedin, %z);
       
       
   local (%colWidth, %forcedin, %merged3cellsWidth, %worksheet, %xlsfmt);

  # Global variables:
  # =================
  # $MAXNMODELS2PRESENT
  # @Dims2Report
  # @KeepValuesAsIsForThese
  # @outcomevars
  # %AddLink
  # %catvarno
  # %col0
  # %forcedin
  # %info
  # %ncatvars
  # %ncol
  # %nmodels
  # %Out
  # %SHEET
  # %valuelabel
  # %xls
  # %xlsfmts
  
  %logMsg = (1 => "repetition of analysis performed on ", 3 => "with following changes:");
  $Out{xls} = $tmpOut{xls};

  use Spreadsheet::WriteExcel;
  $workbook = Spreadsheet::WriteExcel->new($Out{xls});

  foreach $fmtname (keys %xlsfmts)
  {
    $xlsfmt{$fmtname} = $workbook->add_format(%{$xlsfmts{$fmtname}});
  }
  
  foreach $dim (@KeepValuesAsIsForThese)
  {
    $KeepValuesAsIs{$dim} = 1;
    $dim = "nil" . $dim;
    $KeepValuesAsIs{$dim} = 1;
  }

  # Start writing sheets
  
  foreach $outcomevar (@outcomevars)
  {
    undef %isforcedin;
    undef %tmpforcedin;
  
    if (exists $info{$outcomevar}{$FORCEDIN}{0}{1})
    {
      @tmp = keys %{$info{$outcomevar}{$FORCEDIN}{0}};
      foreach (@tmp)
      {
        $var = $info{$outcomevar}{$FORCEDIN}{0}{$_};
        $forcedin{$outcomevar}{$var} = 1;
        $isforcedin{$var} = 1;
      }
      
      foreach (keys %{$info{$outcomevar}{$PROBNE0}{0}})
      {
        $var = $info{$outcomevar}{$VARNAMES}{0}{$_};
        next unless $isforcedin{$var};
        $tmpforcedin{$_} = 1;
      }
    }
    
  
    @tmp = sort {$tmpforcedin{$b} <=> $tmpforcedin{$a} || $info{$outcomevar}{$PROBNE0}{0}{$b} <=> $info{$outcomevar}{$PROBNE0}{0}{$a} || $info{$outcomevar}{$FIRSTMODELIN}{0}{$a} <=> $info{$outcomevar}{$FIRSTMODELIN}{0}{$b}} keys %{$info{$outcomevar}{$PROBNE0}{0}};

    undef @valuenos;
    undef @variables;
    undef @varindices;

    foreach (@tmp)
    {
      $var = $info{$outcomevar}{$VARNAMES}{0}{$_};
      $tmp = $ncol{$outcomevar}{$var} || 1;
      push(@valuenos, 1..$tmp);
      push(@variables, ($var) x $tmp);
      push(@varindices, ($_) x $tmp);
    }

    
    # --- start preparing table ----
    
    ($short, $analysisno) = ($Y{$outcomevar}{short}, $Y{$outcomevar}{no});

    unless (defined $worksheet{$short})
    {
      $worksheet{$short} = $workbook->add_worksheet("$short");
      $z{$short}{row} = 0; # Current xls table row & column
    }
    
    $z{$short}{row} += $xls{sepblankrows} if $z{$short}{row};
    
    $z{$short}{outcomenamerow} = $z{$short}{row}++;
    $z{$short}{inforow}        = $z{$short}{row}++;
    
    $tmp = "n = $info{$outcomevar}{$SAMPLESIZE}{0}{1}";
    $tmp .= " ($info{$outcomevar}{$NEVENTS}{0}{1} events)" if defined $info{$outcomevar}{$NEVENTS}{0}{1};
    &WriteValue($short, $z{$short}{inforow}, 0, "$tmp");
    
    $tmp = &Family($outcomevar);
    &WriteValue($short, $z{$short}{inforow}, 1, "$tmp");
    
    
    @tmp = sort {$a <=> $b} keys %{$info{$outcomevar}{$NOTES}{0}};
    if (@tmp)
    {
      undef $notes;
      foreach (@tmp)
      {
        $notes .= ", " . $info{$outcomevar}{$NOTES}{0}{$_};
      }
      $notes = substr($notes, 2);
     
      $worksheet{$short}->write($z{$short}{row}++, 0, $notes, $xlsfmt{notes});
    }

    
    $z{$short}{indepvarsrow}    = $z{$short}{row}++;
    $z{$short}{firstmodelinrow} = $z{$short}{row}++;
    $z{$short}{probne0row}      = $z{$short}{row}++;
    $z{$short}{forcedinrow}     = $z{$short}{row}++ if exists $forcedin{$outcomevar};
    
    if ($ncatvars{$outcomevar})
    {
      $z{$short}{lblrow}    = $z{$short}{row}++;
      $z{$short}{reflblrow} = $z{$short}{row}++;

      &WriteFormattedValues($short, $z{$short}{lblrow}, 2, "toplbl", "label");
    }
    
    $z{$short}{row} += $xls{afterheaderblanknrows};
    
    
    &WriteFormattedValues($short, $z{$short}{firstmodelinrow}, 2, "toplbl", "First presence in model #");
    &WriteFormattedValues($short, $z{$short}{probne0row}, 2, "toplbl", "probne0");



    $col = $xls{leftblockncolumns};
    @ind = @varindices;
    @val = @valuenos;
    
    
    foreach $var (@variables)
    {
      ($ind, $val) = (shift @ind, shift @val);
      
      $tmp = $info{$outcomevar}{$FIRSTMODELIN}{0}{$ind};
      $inreportedmodels{$outcomevar}{$var} = 1 if $tmp ne "NA" && $tmp <= $MAXNMODELS2PRESENT;
      $nil = $inreportedmodels{$outcomevar}{$var} ? "" : "nil";

      &WriteFormattedValues($short, $z{$short}{indepvarsrow}, $col, $nil . "indepvarname", $var);
      &WriteFormattedValues($short, $z{$short}{firstmodelinrow}, $col, $nil . "firstmodelin", $info{$outcomevar}{$FIRSTMODELIN}{0}{$ind});
       
      $tmp = $info{$outcomevar}{$PROBNE0}{0}{$ind} . "\%";
      &WriteFormattedValues($short, $z{$short}{probne0row}, $col, $nil . "probne0", $tmp);
      &WriteFormattedValues($short, $z{$short}{forcedinrow}, $col, "forcedin") if $forcedin{$outcomevar}{$var};

      if ($ncol{$outcomevar}{$var})
      {
        &WriteFormattedValues($short, $z{$short}{lblrow}, $col, $nil . "catlbl", $valuelabel{$outcomevar}{$var}{$val});
        &WriteFormattedValues($short, $z{$short}{reflblrow}, $col, $nil . "refcatlbl", "ref: $info{$outcomevar}{$REF}{0}{$catvarno{$outcomevar}{$var}}") if $val == 1;
      }

      $tmp = $inreportedmodels{$outcomevar}{$var} ? 3 : 1;
      $col += $tmp;
      $z{$short}{col1} = $col - 1;
    }
    

    @tmp = sort {lc $a cmp lc $b} values %{$info{$outcomevar}{$REJECTEDVARS}{0}};
    if (@tmp)
    {
      $rejected{$short}{col0} = $col;
      @rejected = @tmp;
      
      foreach (@tmp)
      {
        s/\./_/g;
        &WriteFormattedValues($short, $z{$short}{indepvarsrow}, $col, "rejectedvarname", " " . $_ . " ");
        
        $rejected{$short}{col1} = $col++;
      }
      
      $z{$short}{col1} = $rejected{$short}{col1};
    }
    

    $worksheet{$short}->merge_range($z{$short}{outcomenamerow}, 0, $z{$short}{outcomenamerow}, $z{$short}{col1}, $Y{$outcomevar}{displayed}, $xlsfmt{outcomename});
    if ($IsRepetitionOf{$outcomevar})
    {
      # a link will be added to log file, rather (at the stage of log-file writing)
      ($AddLink{$outcomevar}{row}, $AddLink{$outcomevar}{inforow}, $AddLink{$outcomevar}{colspan}) = ($z{$short}{outcomenamerow}, $z{$short}{inforow}, $z{$short}{col1});
    }
    

    # --- Write parameter estimates for each model  ------------------
    
    @modelnos = sort {$a <=> $b} keys %{$info{$outcomevar}{$MODEL}{0}}; 
    @modelnos = &ModelsShortList($MAXNMODELS2PRESENT, @modelnos);
    ($lastmodelno) = reverse @modelnos;
    
    
    foreach $dim (@Dims2Report)
    {
      next unless $AnyInfo{$outcomevar}{$dim};
      
      ($dim0, $dim1) = ("$dim.lower", "$dim.upper");
      
      $z{$short}{sectionheadrow} = $z{$short}{row}++;
      $z{$short}{modelheadrow}   = $z{$short}{row}++;
      $z{$short}{modelrow0}      = $z{$short}{row};

      &WriteFormattedValues($short, $z{$short}{sectionheadrow}, 0, "blocklbl", $Section{$dim});
      &WriteFormattedValues($short, $z{$short}{modelheadrow}, 0, "modelno", "#");
      &WriteValue($short, $z{$short}{modelheadrow}, 1, "Model prob") if $dim eq $MLE;
      &WriteValue($short, $z{$short}{modelheadrow}, 2, "Model");
      
      
      foreach $modelno (@modelnos)
      {
        $tmp = $modelno == $lastmodelno && $lastmodelno < $nmodels{$outcomevar} ? "$lastmodelno (of $nmodels$nmodels{$outcomevar})" : $modelno;
        &WriteFormattedValues($short, $z{$short}{row}, 0, "modelno", $tmp);
        &WriteValue($short, $z{$short}{row}, 2, $info{$outcomevar}{$MODEL}{0}{$modelno});

      
        if ($dim eq $MLE)
        {
          $modelprob = sprintf("%.${MODELPROBNDIGITS}f", $info{$outcomevar}{$MODELPROB}{0}{$modelno}*100) . "\%";
          &WriteValue($short, $z{$short}{row}, 1, $modelprob);
          
          $z{$short}{modelrow1} = $z{$short}{row}
        }
        
        @val = @valuenos;
        $col = $xls{leftblockncolumns};
        
        
        foreach $var (@variables)
        {
          $val = shift @val;
          $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
          $pointestimate = $info{$outcomevar}{$dim}{$modelno}{$parmcol};
          
          
          unless ($pointestimate eq "NA")
          {
            ($lcl, $ucl) = ($info{$outcomevar}{$dim0}{$modelno}{$parmcol},
                            $info{$outcomevar}{$dim1}{$modelno}{$parmcol});

            ($pointestimate, $lcl, $ucl) = &Round($pointestimate, $lcl, $ucl);
            &WriteFormattedValues($short, $z{$short}{row}, $col, 0, $pointestimate, $col + 1, "lcl", $lcl, $col + 2, "ucl", $ucl);
          }
        
          $tmp = $inreportedmodels{$outcomevar}{$var} ? 3 : 1;
          $col += $tmp;
        }
        
        $z{$short}{row}++;
      }
      


      if ($dim eq $MLE)
      {
        # post mean & sd -- label
        
        $z{$short}{row}++;
        $z{$short}{postmomentsrow} = $z{$short}{row}++;
        &WriteFormattedValues($short, $z{$short}{postmomentsrow}, 2, "toplbl", "posterior mean (sd)");

        $z{$short}{priorprobrow} = $z{$short}{row}++;
        &WriteFormattedValues($short, $z{$short}{priorprobrow}, 2, "toplbl", "prior probability");
        
        if ($AnyInfo{$outcomevar}{$FIRSTMODELOUT})
        {
          $z{$short}{firstmodeloutrow} = $z{$short}{row}++;
          &WriteFormattedValues($short, $z{$short}{firstmodeloutrow}, 2, "toplbl", "first model out");
        }

        # post mean & sd -- values

        @val = @valuenos;
        @ind = @varindices;
        $col = $xls{leftblockncolumns};
        
        foreach $var (@variables)
        {
          $nil = $inreportedmodels{$outcomevar}{$var} ? "" : "nil";
          
          ($val, $ind) = (shift @val, shift @ind);
          $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;

          @tmp = ($info{$outcomevar}{$POSTMEAN}{0}{$parmcol}, $info{$outcomevar}{$POSTSD}{0}{$parmcol});          
          @tmp = &Round(@tmp);
          
          &WriteFormattedValues($short, $z{$short}{postmomentsrow}, $col, $nil . "postmoments", @tmp);
          &WriteFormattedValues($short, $z{$short}{priorprobrow}, $col, $nil . "priorprob", $info{$outcomevar}{$PRIORPROB}{0}{$ind});
          
          if (defined $info{$outcomevar}{$PRIORPROB}{0}{$ind})
          {
            $tmp = $info{$outcomevar}{$FIRSTMODELOUT}{0}{$ind};
            $tmp = "--" if $tmp eq "NA";
            &WriteFormattedValues($short, $z{$short}{firstmodeloutrow}, $col, $nil . "firstmodelout", $tmp);
          }
          
          $tmp = $inreportedmodels{$outcomevar}{$var} ? 3 : 1;
          $col += $tmp;
        }


        if (@rejected)
        {
          $tmp = $rejected{$short}{col1} - $rejected{$short}{col0};
          $tmp = $tmp > 0 ? "Rejected variables" : "Rejected";

          $worksheet{$short}->merge_range($z{$short}{firstmodelinrow}, $rejected{$short}{col0}, $z{$short}{modelrow1}, $rejected{$short}{col1}, $tmp, $xlsfmt{rejected});

          $col = $rejected{$short}{col0};
          foreach $var (@rejected)
          {
            &WriteFormattedValues($short, $z{$short}{priorprobrow}, $col++, "rejectedpriorprob", &RejectedPriorProb($outcomevar, $var));
          } 

          delete $rejected{$short};
          undef @rejected;
        }
      }
    }
  }
  
  
  &WriteExcelSummary() if $Write{summary};
  
  
  # --- Add log sheet ----------------------------------------------------------

  $worksheet{$SHEET{log}} = $workbook->add_worksheet($SHEET{log});
  $tmp = "This file was written by $Pkg{name} $Pkg{version}";
  $worksheet{$SHEET{log}}->write(0, 0, $tmp, $xlsfmt{writtenby});
  $tmp = "on " . localtime();
  $worksheet{$SHEET{log}}->write(1, 0, $tmp, $xlsfmt{writtenby});
  
  $row = 3;
  
  foreach $outcomevar (@outcomevars)
  {
    next unless $IsRepetitionOf{$outcomevar};

    #&WriteFormattedValues($SHEET{log}, $row, 0, "logyvar", $Y{$outcomevar}{displayed});
    $xlsrow = $AddLink{$outcomevar}{row} + 1;
    $worksheet{$SHEET{log}}->write($row, 0, "internal:$Y{$outcomevar}{short}!A$xlsrow", $Y{$outcomevar}{displayed}, $xlsfmt{logyvar});    
    $colWidth{$SHEET{log}}{0} = &Max2($colWidth{$SHEET{log}}{0}, length($Y{$outcomevar}{displayed}));
    
    # Write a link to here in outcome spreadsheet, at right cell location
    $xlsrow = $row + 1;
    $worksheet{$Y{$outcomevar}{short}}->write($AddLink{$outcomevar}{inforow}, 2, "internal:$SHEET{log}!A$xlsrow", "details...", $xlsfmt{details}); 
    
    &WriteValue($SHEET{log}, $row, 1, $logMsg{1});
    $row++;
    
    &WriteFormattedValues($SHEET{log}, $row, 1, "logrepfrom", $Y{$IsRepetitionOf{$outcomevar}}{displayed});
    $row++;
    
    &WriteValue($SHEET{log}, $row, 1, $logMsg{3});
    $row++;


    $changes = " " . $changes{$outcomevar};
    $changes =~ s{\\s}{ }g;
    
    while ($changes =~ /(.*)\s(\w+)\>\s+/)
    {
      ($changes, $action, $tmp) = ($1, $2, $');;
      
      if ($action eq "replace")
      {
        $tmp =~ s/^\s+|\s+$//g; # lrtrim
        $tmp =~ s/\s*=\s*/=/g;
        $tmp =~ s/\s+/, /g;
        $tmp =~ s/=/ by /g;
        unshift(@changes, "replaced " . $tmp);
      }
      elsif ($action eq "add")
      {
        $tmp =~ s/^\s+|\s+$//g; # lrtrim
        $tmp =~ s/\s+/, /g;
        unshift(@changes, "added " . $tmp);
      }
      elsif ($action eq "drop")
      {
        $tmp =~ s/^\s+|\s+$//g; # lrtrim
        $tmp =~ s/\s+/, /g;
        unshift(@changes, "dropped " . $tmp);
      }
      elsif ($action eq "yvar")
      {
        $tmp =~ s/^\s+|\s+$//g; # lrtrim
        unshift(@changes, "changed outcome for " . $tmp);
      }
      else
      {
        unshift(@changes, $action . "> " . $tmp);
      }
    }
    unshift(@changes, $changes) if $changes =~ /\S/;
    
    
    foreach $tmp (@changes)
    {
      &WriteFormattedValues($SHEET{log}, $row, 1, "logchange", $tmp);
      $row++;
    }
            
    $row++;
    undef @changes;
  }


  # --- Autofit column width in each sheet ---------------------------------------------------

  foreach $short (keys %worksheet)
  {
    # First widen individual columns width to adjust for width of corresponding merged columns
    foreach $c (keys %{$merged3cellsWidth{$short}})
    {
      @k = ($c, $c + 1, $c + 2);
      $tmp = $merged3cellsWidth{$short}{$c};
      foreach $k (@k)
      {
        $tmp -= $colWidth{$short}{$k};
      }
      next if $tmp <= 0;
      $modulo = $tmp%3;
      $tmp += 3 - $modulo if $modulo;
      $tmp /= 3;
      foreach $k (@k)
      {
        $colWidth{$short}{$k} += $tmp;
      }
    }

    # Set column width
    foreach $col (keys %{$colWidth{$short}})
    {
      $width = $colWidth{$short}{$col};
      next unless $width;
      $worksheet{$short}->set_column($col, $col, $width);
    }
  }
  
    
  $workbook->close();

} # end of WriteExcelFile


sub WriteExcelSummary()
{
  my ($analysisno, $col, $indepvar, $modelno, $modelprob,
      $outcomevar, $row, $short, $toprow, $var, $varindex,
      @analysesnos, @indepvars, @short, @tmp,
      %firstanalysisno, %is_indepvar, %maxprob,
      %outcomevar, %probne0, %rejected);

  # Global:
  # -------
  # @outcomevars
  # %forcedin
  # %info
  # %SHEET
  # %Y


  $worksheet{$SHEET{summary}} = $workbook->add_worksheet($SHEET{summary});
  $row = 0;


  foreach $outcomevar (@outcomevars)
  {
    ($short, $analysisno) = ($Y{$outcomevar}{short}, $Y{$outcomevar}{no});
    $outcomevar{$short}{$analysisno} = $outcomevar;
  }
  
  @short = sort {&MixedOrder($a, $b)} keys %outcomevar;
  
  
  foreach $short (@short)
  {
    undef %firstanalysisno;
    undef %is_indepvar;
    undef %maxprob;
    undef %probne0;
    undef %rejected;
    
    $toprow = $row;
    $row += 3;
    
    $tmp = "Best $NBESTMODELSPROBS Models Probabilities";
    &WriteFormattedValues($SHEET{summary}, $toprow + 1, 0, "summarylbl", $tmp);
    &WriteFormattedValues($SHEET{summary}, $toprow + 1, 1, "summarylbl", "Best Model");
    &WriteFormattedValues($SHEET{summary}, $toprow + 1, 2, "summarynolbl", "Analysis no");

  
    @analysesnos = sort {$a <=> $b} keys %{$outcomevar{$short}};
  
    foreach $analysisno (@analysesnos)
    {
      $outcomevar = $outcomevar{$short}{$analysisno};
      @tmp = values %{$info{$outcomevar}{$REJECTEDVARS}{0}};
      foreach $var (@tmp)
      {
        $is_indepvar{$var} = 1;
        $firstanalysisno{$var} = $analysisno unless exists $firstanalysisno{$var};

        $rejected{$var}{$analysisno} = 1;
      }


      @tmp = keys %{$info{$outcomevar}{$VARNAMES}{0}};
      foreach $varindex (@tmp)
      {
        $var = $info{$outcomevar}{$VARNAMES}{0}{$varindex};
        $is_indepvar{$var} = 1;
        $firstanalysisno{$var} = $analysisno unless exists $firstanalysisno{$var};

        $probne0{$var}{$analysisno} = $info{$outcomevar}{$PROBNE0}{0}{$varindex};
        $maxprob{$var} = &Max2($maxprob{$var}, $probne0{$var}{$analysisno});
      }
      
      
      # Best models probs
      undef @tmp;
      foreach $modelno (1..$NBESTMODELSPROBS)
      {
        last unless exists $info{$outcomevar}{$MODELPROB}{0}{$modelno};
        $modelprob = sprintf("%.${MODELPROBNDIGITS}f", $info{$outcomevar}{$MODELPROB}{0}{$modelno}*100);
        push(@tmp, $modelprob);
      }
      $tmp = join(", ", @tmp);
      &WriteValue($SHEET{summary}, $row, 0, $tmp);
      
      # Best model (variables)
      &WriteValue($SHEET{summary}, $row, 1, $info{$outcomevar}{$MODEL}{0}{1});
      
      # Analysis no
      &WriteFormattedValues($SHEET{summary}, $row, 2, "analysisno", "\[" . $analysisno . "\]");

      $row++;
    }
          

    @indepvars = sort {$maxprob{$b} <=> $maxprob{$a} || $firstanalysisno{$a} <=> $firstanalysisno{$b}} keys %is_indepvar;
    $col = 3;

    foreach $indepvar (@indepvars)
    {
      $row = $toprow + 2;
      &WriteFormattedValues($SHEET{summary}, $row, $col, "summaryindep", " " . $indepvar);
      $row++;
      
      foreach $analysisno (@analysesnos)
      {
        $outcomevar = $outcomevar{$short}{$analysisno};
        $tmp = &WhichVarIndex($indepvar, %{$info{$outcomevar}{$VARNAMES}{0}});
        
        if ($tmp < 0)
        {
          $tmp = $rejected{$indepvar}{$analysisno} ? "rejected" : "n/a";
          &WriteFormattedValues($SHEET{summary}, $row, $col, "summarycomment", $tmp);
        }
        else
        {
          $tmp = $info{$outcomevar}{$PROBNE0}{0}{$tmp};
          $tmp .= " (forced in) " if $forcedin{$outcomevar}{$ivar};
          &WriteValue($SHEET{summary}, $row, $col, $tmp);
        }
      
        $row++;
      }
      
      $col++;
    }
    
    $col = 3 + $#indepvars;
    $worksheet{$SHEET{summary}}->merge_range($toprow + 1, 3, $toprow + 1, $col, "Independent variables", $xlsfmt{summarylblindep});
    $worksheet{$SHEET{summary}}->merge_range($toprow, 0, $toprow, $col, $short, $xlsfmt{outcomename});
    
    $row += 3;
  }
} # end of WriteExcelSummary


sub WriteFormattedValues()
{
  my ($short, $row, @etc) = @_;
  my ($col, $dim, $l, $m, $value,
      @values);
      
  # Global variables
  # ----------------
  # %colspan
  # %merged3cellsWidth (out)
  # %worksheet
  # %xlsfmt
  
  while (@etc)
  {
    ($col, $dim, @etc) = @etc;
    $dim = lc $dim;

    $m = $dim eq "postmoments" || $dim eq "nilpostmoments" ? 2 : 1;
    @values = splice(@etc, 0, $m);
    
    unless ($KeepValuesAsIs{$dim})
    {
      foreach $value (@values)
      {
        $value = &NotScientific($value);
      }
      
      $values[1] = "(" . $values[1] . ")" if $dim eq "postmoments" || $dim eq "nilpostmoments";
    }
    
    if ($dim eq "forcedin")
    {
      $value = "(forced in)";
    }
    else
    {
      $value = join(" ", @values);
    }
    
    
    if ($dim eq "lcl")
    {
      $value = "(" . $value . ",";
    }
    elsif ($dim eq "ucl")
    {
      $value = " " . $value . ")";
    }
    
    
    $l = length($value);
    
    if ($colspan{$dim} == 3)
    {
      $worksheet{$short}->merge_range($row, $col, $row, $col+2, $value, $xlsfmt{$dim});
      $merged3cellsWidth{$short}{$col} = &Max2($merged3cellsWidth{$short}{$col}, $l);
    }
    else
    {      
      $worksheet{$short}->write($row, $col, $value, $xlsfmt{$dim});
      $colWidth{$short}{$col} = &Max2($colWidth{$short}{$col}, $l);
    }
  }
} # end of WriteFormattedValues


sub WriteHtmlFile()
{
  my ($analysisno, $c, $changes, $class, $color, $colspan, $contd, $continued, $dim, $dim0, $dim1,
      $f, $firstpass, $forcedin, $i, $ind, $indepvar, $ivar, 
      $k, $lcl, $leftcolspan, $leftwidth, $MaxNCols, $modelno, $modelprob, $ncols, $notes, $nsame, 
      $outcomevar, $parmcol, $pointestimate, 
      $short, $showModelProb, $tablewidth, $tmp, $ucl, $val, $var, $varindex, $yvar,
      @analysesnos, @html, @htmlhead, @ind, @indepvars, @ivars,
      @modelnos,
      @otherindices, @otherindices0, @othervaluenos, @othervaluenos0, @othervariables, @othervariables0,
      @short, @tmp,
      @val, @valuenos, @valuenos0, @var, @variables, @variables0, @varindices, @varindices0,
      %firstanalysisno, %forcedin, %isforcedin, %is_indepvar, %maxprob, %outcomevar, %probne0, %rejected, %tmpforcedin);
  
  # Global variables:
  # -----------------
  # @Dims2Report
  # @LEFTWIDTH
  # @outcomevars
  # @SUMCOLORS
  # %col0
  # %info
  # %MAXWIDTH
  # %ncatvars
  # %PaperFmt
  # %Repeated
  # %Style
  # %Y
  # $AVOIDEMPTYTABLES
  # $BLANK
  # $LEFTCOLSPAN
  # $MAXNVARSPERBLOCK
  # $MLE
  # $VARCOLSPAN
  # $VARWIDTH
  
  $Out{html} = $tmpOut{html};
  

  @html = ("<html>");
  
  # --- Style sheet ------------------------------------
  
  push(@html, "<head><style type=\"text/css\">");
  
  foreach (keys %Style)
  {
    push(@html, "$_\{", "  " . $Style{$_}, "\}");
  }
  
  push(@html, "</style></head>");
  
  
  # ----------------------------------------------------
  
  push(@html, "<body>");
  @htmlhead = @html; # make a copy for summary html file
  
  
  foreach $outcomevar (@outcomevars)
  {
    @tmp = values %{$info{$outcomevar}{$REJECTEDVARS}{0}};
    foreach (@tmp)
    {
      s/_/\./g;
    }
    
    undef %isforcedin;
    undef %tmpforcedin;
  
    @tmp = keys %{$info{$outcomevar}{$FORCEDIN}{0}};
    if (@tmp)
    {
      foreach (@tmp)
      {
        $var = $info{$outcomevar}{$FORCEDIN}{0}{$_};
        $forcedin{$outcomevar}{$var} = 1;
        $isforcedin{$var} = 1;
      }
      
      foreach (keys %{$info{$outcomevar}{$PROBNE0}{0}})
      {
        $var = $info{$outcomevar}{$VARNAMES}{0}{$_};
        next unless $isforcedin{$var};
        $tmpforcedin{$_} = 1;
      }
    }
    
  
    # ---
  
    $continued = 0;
    @varindices = sort {$tmpforcedin{$b} <=> $tmpforcedin{$a} || $info{$outcomevar}{$PROBNE0}{0}{$b} <=> $info{$outcomevar}{$PROBNE0}{0}{$a} || $info{$outcomevar}{$FIRSTMODELIN}{0}{$a} <=> $info{$outcomevar}{$FIRSTMODELIN}{0}{$b}} keys %{$info{$outcomevar}{$PROBNE0}{0}};
  
    undef @valuenos0;
    undef @variables0;
    undef @varindices0;
  
  
    foreach (@varindices)
    {
      $var = $info{$outcomevar}{$VARNAMES}{0}{$_};
      $tmp = $ncol{$outcomevar}{$var};
      $tmp ||= 1;
      push(@valuenos0, 1..$tmp);
      push(@variables0, ($var) x $tmp);
      push(@varindices0, ($_) x $tmp);
    }
  
  
    # --- drop last columns of table if doing so can avoid empty tables (if requested)
  
    if ($AVOIDEMPTYTABLES)
    {
      undef @otherindices0;
  
      $i = $#variables0;
      while ($i >= 0)
      {
        $ind = $varindices0[$i];
        $f = $info{$outcomevar}{$FIRSTMODELIN}{0}{$ind};
        last if $f ne "NA" && $f <= $MAXNMODELS2PRESENT;
  
        unshift(@otherindices0, $ind);
        $i--;
      }
  
      # cut in columns only if it is worth it (only if it saves one or more subtable)
  
      if (@otherindices0 && ($#otherindices0+1) >= ($#variables0+1)%$MAXNVARSPERBLOCK{$PaperFmt{main}})
      {
        $k = @otherindices0;
        splice(@varindices0, -$k);
  
        @othervariables0 = splice(@variables0, -$k);
        @othervaluenos0  = splice(@valuenos0, -$k);
      }
      else
      {
        undef @otherindices0;
      }
    }
  
    # --- start preparing tables ---
    
    
    ($short, $analysisno) = ($Y{$outcomevar}{short}, $Y{$outcomevar}{no}); 
    $outcomevar{$short}{$analysisno} = $outcomevar;
      
  
    while (@valuenos0)
    {
      $contd = $continued ? "\&nbsp;\&nbsp;\&nbsp;(cont'd)" : "";
      $leftcolspan = $continued ? 1 : $LEFTCOLSPAN;
  
      @valuenos   = splice(@valuenos0,   0, $MAXNVARSPERBLOCK{$PaperFmt{main}});
      @variables  = splice(@variables0,  0, $MAXNVARSPERBLOCK{$PaperFmt{main}});
      @varindices = splice(@varindices0, 0, $MAXNVARSPERBLOCK{$PaperFmt{main}}); 
  
      $tablewidth = $LEFTWIDTH[$continued] + $VARWIDTH * @valuenos;
      $tablewidth = $MAXWIDTH{$PaperFmt{main}} if $tablewidth > $MAXWIDTH{$PaperFmt{main}};
      $tablewidth = $tablewidth ? "width=$tablewidth" : "";
      push(@html, "<a name=\"$outcomevar\"></a>") if $Repeated{$outcomevar};
      push(@html, "<table border=0 cellpadding=0 cellspacing=0 class=bic $tablewidth>");
      push(@html, "<tr>");
  
      $tmp = $Y{$outcomevar}{html};
      $tmp = "<a href=\"$Out{htmlsummaryfname}#$short\"> $tmp </a>" if $Write{summary};
      $tmp = $contd if $continued;
      push(@html, "<td colspan=$leftcolspan class=$TitleClass[$continued]> $tmp </td>");
  
      @tmp = @variables;
      
      while (@tmp)
      {
        $tmp = shift @tmp;
        $nsame = &NSame($tmp, @tmp);
        $colspan = $VARCOLSPAN * ($nsame + 1);
        push(@html, "<td colspan=$colspan class=title> $tmp </td>");
        splice(@tmp, 0, $nsame) if $nsame;
      }
      push(@html, "</tr>");
  
      push(@html, "<tr><td valign=top class=family colspan=$leftcolspan> ");
      $tmp = &Family($outcomevar);
      $tmp = "&nbsp;" if $continued;
      push(@html, "$tmp </td>");
  
  
      @val = @valuenos;
      @ind = @varindices;
      @var = @variables;
      
      while (@ind)
      {
        ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
        $class = ($val < $ncol{$outcomevar}{$var}) ? "firstin4cat" : "firstin";
      
        if ($val == 1)
        {
          $tmp = "first presence in model # " . $info{$outcomevar}{$FIRSTMODELIN}{0}{$ind};
        }
        else
        {
          $tmp = $BLANK;
        }
        
        push(@html, "<td valign=top class=$class colspan=$VARCOLSPAN> $tmp </td>");
      }
    
      push(@html, "</tr>");
  
  
      push(@html, "<tr>");
      $tmp = $leftcolspan;
      unless ($continued)
      {
        undef @tmp;
        $tmp--;
        
        push(@tmp, "n = $info{$outcomevar}{$SAMPLESIZE}{0}{1}");
        push(@tmp, "($info{$outcomevar}{$NEVENTS}{0}{1} events)") if defined $info{$outcomevar}{$NEVENTS}{0}{1}; 
        
        push(@html, "<td valign=top class=n colspan=$tmp>" . join("\&nbsp;\&nbsp", @tmp) . "</td>");
              
        $tmp = 1;
      }
      push(@html, "<td valign=top class=infocell0 colspan=$tmp> probne0 </td>");
  
      @val = @valuenos;
      @ind = @varindices;
      @var = @variables;
      
      while (@ind)
      {
        ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
  
        $tmp = ($val == 1) ? $info{$outcomevar}{$PROBNE0}{0}{$ind} : "";
        push(@html, "<td valign=top class=leftcell> $tmp </td>");
        
        $tmp = $VARCOLSPAN - 1;
        $forcedin = $forcedin{$outcomevar}{$var} ? "(forced in)" : $BLANK;
        $class = ($val < $ncol{$outcomevar}{$var}) ? "forcedin4cat" : "forcedin";
        push(@html, "<td valign=top class=$class colspan=$tmp> $forcedin </td>");
      }
      push(@html, "</tr>");
  
      # --- Categorical variables: print value labels (and reference value label)
  
      if ($ncatvars{$outcomevar} || (!$continued && $k{$outcomevar}{$NOTES}{0}))
      {
        push(@html, "<tr>");
  
        $tmp = $leftcolspan;
  
        if (!$continued && $k{$outcomevar}{$NOTES}{0})
        {
          @tmp = sort {$a <=> $b} keys %{$info{$outcomevar}{$NOTES}{0}};
          undef $notes;
          foreach (@tmp)
          {
            $notes .= ", " . $info{$outcomevar}{$NOTES}{0}{$_};
          }
          $notes = substr($notes, 2);
  
          $class = "notes";
          ($tmp, $class) = ($tmp - 1, "notesandlabels") if $ncatvars{$outcomevar};
  
          push(@html, "<td valign=top class=$class colspan=$tmp> $notes </td>");
          $tmp = 1;
        }
              
  
        if ($ncatvars{$outcomevar})
        {        
          # -- value labels
  
          push(@html, "<td valign=top class=infocell0 colspan=$tmp> label </td>");
  
          @val = @valuenos;
          @ind = @varindices;
          @var = @variables;
  
          while (@ind)
          {
            ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
            
            $tmp = $ncol{$outcomevar}{$var} ? $valuelabel{$outcomevar}{$var}{$val} : $BLANK; 
  
            $class = ($val == $ncol{$outcomevar}{$var} || !$ncol{$outcomevar}{$var}) ? "valuelabel1" : "valuelabel";
            push(@html, "<td valign=top colspan=$VARCOLSPAN class=$class> $tmp </td>");
          }
          push(@html, "</tr>");
  
          # -- reference value labels
  
          push(@html, "<tr><td valign=top class=infocell0 colspan=$leftcolspan> $BLANK </td>");
  
          @val = @valuenos;
          @ind = @varindices;
          @var = @variables;
  
          while (@ind)
          {
            ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
  
            $tmp = $ncol{$outcomevar}{$var} && $val == 1 ? "ref: $info{$outcomevar}{$REF}{0}{$catvarno{$outcomevar}{$var}}" : $BLANK;          
            $class = ($val == $ncol{$outcomevar}{$var} || !$ncol{$outcomevar}{$var}) ? "refvaluelabel1" : "refvaluelabel";
            push(@html, "<td valign=top colspan=$VARCOLSPAN class=$class> $tmp </td>");
          }
        }
  
        push(@html, "</tr>");
      }
  
  
      # ---- table body (parms estimates and ci's)
  
      $showModelProb = 1;
      foreach $dim (@Dims2Report)
      {
        next unless $AnyInfo{$outcomevar}{$dim};
  
        $tmp = $continued ? 1 + $VARCOLSPAN*@varindices : $LEFTCOLSPAN;
        push(@html, "<tr class=section><td colspan=$tmp class=subtitlecell0> $Section{$dim}$contd</td>");
        push(@html, "</tr>");
  
        @modelnos = sort {$a <=> $b} keys %{$info{$outcomevar}{$MODEL}{0}};
        @modelnos = &ModelsShortList($MAXNMODELS2PRESENT, @modelnos);
        ($dim0, $dim1) = ("$dim.lower", "$dim.upper");
  
        push(@html, "<tr><td valign=top align=right>#</td>");
        unless ($continued)
        {
          push(@html, "<td valing=top align=right>");
          push(@html, "<i>Model<br>prob</i>") if $showModelProb;
          push(@html, "</td><td valign=top class=infocell0>Model</td>");
        }
        push(@html, "</tr>");
  
        $color = 0;
        
        foreach $modelno (@modelnos)
        {         
          undef $modelprob;
          $modelprob = sprintf("%.${MODELPROBNDIGITS}f", $info{$outcomevar}{$MODELPROB}{0}{$modelno}*100) . "\%" if $showModelProb;
          
          push(@html, "<tr bgcolor=$COLORS[$color]>");
          $tmp = ($modelno == $modelnos[$#modelnos] && $modelno < $nmodels{$outcomevar}) ? "(of $nmodels{$outcomevar}) " : "";
          push(@html, "<td valign=top class=$ModelNoClass[$continued]>$tmp$modelno</td>");
          push(@html, "<td valign=top align=right> $modelprob </td>") unless $continued;
        
          push(@html, "<td valign=top class=modellabel width=$LABELWIDTH> $info{$outcomevar}{$MODEL}{0}{$modelno}</td>") unless $continued;
  
          @val = @valuenos;
          @var = @variables;
  
          while (@var)
          {
            ($val, $var) = (shift @val, shift @var);
  
            $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
            $pointestimate = $info{$outcomevar}{$dim}{$modelno}{$parmcol};
            if ($pointestimate eq "NA")
            {
              push(@html, "<td class=uppercl colspan=$VARCOLSPAN>&nbsp;</td>");
            }
            else
            {
              ($lcl, $ucl) = ($info{$outcomevar}{$dim0}{$modelno}{$parmcol},
                              $info{$outcomevar}{$dim1}{$modelno}{$parmcol});
  
              ($pointestimate, $lcl, $ucl) = &Round($pointestimate, $lcl, $ucl);
  
              push(@html, "<td valign=top class=pointestimate>$pointestimate</td>");
              push(@html, "<td valign=top align=right> ($lcl, </td>");
              push(@html, "<td valign=top class=uppercl> &nbsp;&nbsp; $ucl)</td>");           
            }
          }
  
          push(@html, "</tr>");
          $color = 1 - $color;
        }
  
        $showModelProb = 0;
  
        # ---
  
        if ($dim eq $MLE)
        {
          # -- post mean & sd
  
          push(@html, "<tr class=postmoments>");
          push(@html, "<td class=infocell0postm colspan=$leftcolspan> posterior mean (sd)</td>");
    
          @val = @valuenos;
          @var = @variables;
          @ind = @varindices;
          
            
          while (@var)
          {
            ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
            $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
  
            @tmp = ($info{$outcomevar}{$POSTMEAN}{0}{$parmcol}, $info{$outcomevar}{$POSTSD}{0}{$parmcol});          
            @tmp = &Round(@tmp);
  
            push(@html, "<td valign=top class=pointestimatepostm> $tmp[0] </td>");
            push(@html, "<td valign=top class=postm> ($tmp[1])</td>");
            push(@html, "<td class=postrcell0>$BLANK</td>");
          }
          push(@html, "</tr>");
          
  
          # -- prior prob
  
          push(@html, "<tr class=priorprob>");
          push(@html, "<td class=infocell0 colspan=$leftcolspan> prior probability</td>");
    
          @val = @valuenos;
          @var = @variables;
          @ind = @varindices;
          
          while (@var)
          {
            ($val, $ind, $var) = (shift @val, shift @ind, shift @var);
            $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
  
            $tmp = ($val == 1) ? $info{$outcomevar}{$PRIORPROB}{0}{$ind} : $BLANK;
            push(@html, "<td class=postrcell colspan=$VARCOLSPAN> $tmp </td>");
          }
          push(@html, "</tr>");
  
          # -- rejected vars        
  
          if (!$continued && $k{$outcomevar}{$REJECTEDVARS}{0})
          {
            @tmp = sort {lc $a cmp lc $b} values %{$info{$outcomevar}{$REJECTEDVARS}{0}};
            push(@html, "<tr class=rejectedvars>");
            push(@html, "<td class=infocell0 valign=top colspan=$LEFTCOLSPAN> rejected variables </td>");
            $tmp = $VARCOLSPAN * @varindices;
            push(@html, "<td colspan=$tmp valign=top class=rejectedvarslist>");
            $tmp = join(", ", @tmp);
            push(@html, " $tmp </td></tr>");
          }
  
          # --- other variables ---
  
          if (@otherindices0 && !$continued)
          {
            $tmp = $VARCOLSPAN * @varindices + $LEFTCOLSPAN;
            push(@html, "<tr><td colspan=$tmp align=$OTHERVARSALIGN><table cellspacing=0 cellpadding=0>");
  
            # -- header (top line) --
  
            @otherindices   = @otherindices0;
            @othervariables = @othervariables0;
            @othervaluenos  = @othervaluenos0;
  
            push(@html, "<tr><td class=othervarsheader0 valign=top> Other variables </td>");
            
            while (@othervariables)
            {
              ($var, $val) = (shift @othervariables, shift @othervaluenos);
  
              $class = $val < $ncol{$outcomevar}{$var} ? "othervarscatheader" : "othervarsheader";
              push(@html, "<td class=$class valign=top> $var");
              push(@html, "<br> $valuelabel{$outcomevar}{$var}{$val}") if $ncol{$outcomevar}{$var};
              push(@html, "</td>");
            }
            push(@html, "</tr>");
  
            # -- probne0 [priorprob] --
  
            @otherindices   = @otherindices0;
            @othervariables = @othervariables0;
            @othervaluenos  = @othervaluenos0;
  
            push(@html, "<tr><td class=othervarsleft valign=top> probne0 [prior probability] </td>");
            
            while (@otherindices)
            {
              ($ind, $var, $val) = (shift @otherindices, shift @othervariables, shift @othervaluenos);
  
              $class = $val < $ncol{$outcomevar}{$var} ? "othervarscatentry" : "othervarsentry";
              $tmp = $val == 1 ? "$info{$outcomevar}{$PROBNE0}{0}{$ind} [$info{$outcomevar}{$PRIORPROB}{0}{$ind}]" : $BLANK;
              push(@html, "<td class=$class valign=top> $tmp </td>");
            }
            push(@html, "</tr>");
  
            # -- first model in --
  
            @otherindices   = @otherindices0;
            @othervariables = @othervariables0;
            @othervaluenos  = @othervaluenos0;
  
            push(@html, "<tr><td class=othervarsleft valign=top> first presence in model # </td>");
            
            while (@otherindices)
            {
              ($ind, $var, $val) = (shift @otherindices, shift @othervariables, shift @othervaluenos);
  
              $class = $val < $ncol{$outcomevar}{$var} ? "othervarscatentry" : "othervarsentry";
              $tmp = $val == 1 ? $info{$outcomevar}{$FIRSTMODELIN}{0}{$ind} : $BLANK;
              push(@html, "<td class=$class valign=top> $tmp </td>");
            }
            push(@html, "</tr>");
  
            # -- posterior mean --
  
            @othervariables = @othervariables0;
            @othervaluenos  = @othervaluenos0;
  
            push(@html, "<tr><td class=othervarsleft valign=top> posterior mean </td>");
            
            while (@othervariables)
            {
              ($var, $val) = (shift @othervariables, shift @othervaluenos);
  
              $class = $val < $ncol{$outcomevar}{$var} ? "othervarscatentry" : "othervarsentry";
              $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
              ($tmp) = &Round($info{$outcomevar}{$POSTMEAN}{0}{$parmcol});
              push(@html, "<td class=$class valign=top> $tmp </td>");
            }
            push(@html, "</tr>");
  
            # -- posterior sd --
  
            @othervariables = @othervariables0;
            @othervaluenos  = @othervaluenos0;
  
            push(@html, "<tr><td class=othervarsleftlast valign=top> posterior sd </td>");
            
            while (@othervariables)
            {
              ($var, $val) = (shift @othervariables, shift @othervaluenos);
  
              $class = "othervarsbottomentry";
              $parmcol = $col0{$outcomevar}{$dim}{$var} + $val - 1;
              ($tmp) = &Round($info{$outcomevar}{$POSTSD}{0}{$parmcol});
              push(@html, "<td class=$class valign=top> $tmp </td>");
            }
            push(@html, "</tr>");
  
            push(@html, "</table></td></tr>");
          }
          
          if ($IsRepetitionOf{$outcomevar})
          { 
            $tmp = $Y{$IsRepetitionOf{$outcomevar}}{html};
            
            # Replace (add|replace|drop|etc)> by a solid dark triangle followed by the keyword
            $changes = $changes{$outcomevar};
            $changes =~ s{\\s}{ }g;
            $changes = $` . "&#9658;" . " $1: " . $' while $changes =~ /(\w+)\>\s/;
            
            push(@html, "<td><td>&nbsp;</td></tr>",
                        "<tr><td class=insight colspan=4>Repetition of analysis performed on <a href=\"#$IsRepetitionOf{$outcomevar}\" class=insight>$tmp</a> with following changes:<br><br>",
                        "<span class=insight>$changes</span></td></tr>");
          }
        }  
      }
  
      push(@html, "<tr><td>&nbsp;</td></tr></table><br>");
      $continued = 1;
    }
  }
  
  
  push(@html, "</table></body></html>");
  
  
  # ---- Write html file ---------------
  
  open(HTML, ">$Out{html}") || die "Cannot write output to file $Out{html}. Abort.\n";
  foreach (@html)
  {
    print HTML $_ . "\n";
  }
  close HTML;
   
  
  # ===== Prepare Summary html file =======================================================
  
  
  if ($Write{summary})
  {
    $Out{htmlsummary} = $tmpOut{htmlsummary};
    open(SUMMARY, ">$Out{htmlsummary}") || die "Cannot write to file $Out{htmlsummary}. Abort.\n";
      
  
    $leftwidth = $BESTMODELWIDTH + $BESTMODELPROBSWIDTH + $ANALYSISNOWIDTH;
    $MaxNCols = int (($MAXWIDTH{$PaperFmt{summary}} - $leftwidth)/$PROBNE0WIDTH);
  
    @html = @htmlhead;
    @short = sort {&MixedOrder($a, $b)} keys %outcomevar;
  
    foreach $short (@short)
    {
      # --- some cleaning first ---------------------------------
  
      undef %firstanalysisno;
      undef %is_indepvar;
      undef %maxprob;
      undef %probne0;
      undef %rejected;
  
      # ---------------------------------------------------------
  
      @analysesnos = sort {$a <=> $b} keys %{$outcomevar{$short}};
  
      foreach $analysisno (@analysesnos)
      {
        $outcomevar = $outcomevar{$short}{$analysisno};
        @tmp = values %{$info{$outcomevar}{$REJECTEDVARS}{0}};
        foreach $var (@tmp)
        {
          $is_indepvar{$var} = 1;
          $firstanalysisno{$var} = $analysisno unless exists $firstanalysisno{$var};
  
          $rejected{$var}{$analysisno} = 1;
        }
  
        @tmp = keys %{$info{$outcomevar}{$VARNAMES}{0}};
        foreach $varindex (@tmp)
        {
          $var = $info{$outcomevar}{$VARNAMES}{0}{$varindex};
          $is_indepvar{$var} = 1;
          $firstanalysisno{$var} = $analysisno unless exists $firstanalysisno{$var};
  
          $probne0{$var}{$analysisno} = $info{$outcomevar}{$PROBNE0}{0}{$varindex};
          $maxprob{$var} = &Max2($maxprob{$var}, $probne0{$var}{$analysisno}); 
        }
      }
  
      @indepvars = sort {$maxprob{$b} <=> $maxprob{$a} || $firstanalysisno{$a} <=> $firstanalysisno{$b}} keys %is_indepvar;
  
      push(@html, "<a name=\"$short\">");
      push(@html, "<table cellspacing=0 cellpadding=0>");
  
      $firstpass = 1;
      while (@indepvars)
      {
        @ivars = splice(@indepvars, 0, $MaxNCols);
  
        if ($firstpass)
        {
          $ncols = @ivars;
  
          push(@html, "<tr class=probne0title><td class=probne0outcomevarheader colspan=3>" . $Y{$outcomevar}{html} . "</td>");
          push(@html, "<td class=probne0indepvarsheader colspan=$ncols> Independent variables </td></tr>");
  
          push(@html, "<tr class=probne0subtitle><td class=probne0subtitle valign=bottom> Best $NBESTMODELSPROBS Models <br> Probabilities </td>");
          push(@html, "<td class=probne0subtitle valign=top> Best Model </td>");
          push(@html, "<td class=probne0subtitleanalysisno valign=bottom> Analysis <br> no. </td>");
        }
        else
        {
          push(@html, "<tr>" . "<td bgcolor=white>$BLANK</td>" x 2);
          push(@html, "<td class=probne0subtitleanalysisno valign=bottom> $BLANK </td>");
        }
  
        foreach $ivar (@ivars)
        {
          push(@html, "<td class=probne0varname valign=bottom width=$PROBNE0WIDTH> $ivar </td>");
        }
  
        push(@html, "</tr>");
  
        $c = 0;
        foreach $analysisno (@analysesnos)
        {
          $outcomevar = $outcomevar{$short}{$analysisno};
  
          $color = $SUMCOLORS[$c];
          push(@html, "<tr bgcolor=$color>");
  
          $class = $firstpass ? "probne0bestmodel" : "probne0bestmodelempty";
          if ($firstpass)
          {
            undef @tmp;
            foreach $modelno (1..$NBESTMODELSPROBS)
            {
              last unless exists $info{$outcomevar}{$MODELPROB}{0}{$modelno};
              $modelprob = sprintf("%.${MODELPROBNDIGITS}f", $info{$outcomevar}{$MODELPROB}{0}{$modelno}*100);
              push(@tmp, $modelprob);
            }
            $tmp = join(", ", @tmp);
          }
          else
          {
            $tmp = $BLANK;
          }
          push(@html, "<td class=$class width=$BESTMODELPROBSWIDTH> $tmp </td>");
  
          $tmp = $firstpass ? $info{$outcomevar}{$MODEL}{0}{1} : $BLANK;
          push(@html, "<td class=$class width=$BESTMODELWIDTH> $tmp </td>");
  
          push(@html, "<td class=probne0analysisno valign=bottom> [$analysisno] </td>");
  
          foreach $ivar (@ivars)
          {
            $class = "probne0value";
            $tmp = &WhichVarIndex($ivar, %{$info{$outcomevar}{$VARNAMES}{0}});
  
            if ($tmp < 0)
            {
              $class = "probne0rejected";
              $tmp   = $rejected{$ivar}{$analysisno} ? "rejected" : "n/a";
            }
            else
            {
              $tmp = $info{$outcomevar}{$PROBNE0}{0}{$tmp};
              $tmp .= "<sup>f</sup>" if $forcedin{$outcomevar}{$ivar};
            }
  
            push(@html, "<td class=$class> $tmp </td>");
          }
  
          push(@html, "</tr>");
          $c = 1 - $c;
        }
  
        $firstpass = 0;
      }
  
      push(@html, "</table><br><br>");
    }
  
    push(@html, "</body></html>");
  
  
    # ---- Write Summary html file --------------------------------------------------
  
    foreach (@html)
    {
      print SUMMARY $_ . "\n";
    }
  
    close SUMMARY;
  }
} # end of WriteHtmlFile


sub WriteValue()
{
  my ($short, $row, $col, $value) = @_;
  my ($l);
  
  # Global values:
  # --------------
  #
  # in:
  # %worksheet
  #
  # out:
  # %colWidth
  
  $worksheet{$short}->write($row, $col, $value);
  $l = length($value);
  $colWidth{$short}{$col} = &Max2($colWidth{$short}{$col}, $l);
} # end of WriteValue
