# Consensus-Based Sample Size - Normal mean
#   Version 0.11 (January 2015)


ss.cons.norm.knownvar.avg.hpdlimits <- function(accepted.diff, prec, 
                                                          prior1, prior2, prior1.mixture.wt=0.5, 
                                                          level=0.95, n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.hpdlimits(accepted.diff, prec,
                                                          prior1, prior2, 
                                                          level, n.max=n.max, prior1.mixture.wt=prior1.mixture.wt)
  }
  

  out <- c(out, list(accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2,
                     prior1.mixture.wt=prior1.mixture.wt, level=level, n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.avg.hpdlimits


ss.cons.norm.knownvar.avg.hpdlimits.mymarg <- function(accepted.diff, prec,
                                                                 prior1, prior2, clinical.prior,
                                                                 level=0.95, n.max=100000)
{
  # All of prior1, prior2 and clinical.prior must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)
  
  
  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.hpdlimits(accepted.diff, prec,
                                                          prior1, prior2, 
                                                          level, n.max=n.max, 
                                                          clinical.prior=clinical.prior)
  }


  out <- c(out, list(accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2, 
                     clinical.prior=clinical.prior, level=level, n.max=n.max, exact.results=T))
                     
  return(out)
} # end of ss.cons.norm.knownvar.avg.hpdlimits.mymarg


ss.cons.norm.knownvar.avg.hpdlimits.bothmarg <- function(accepted.diff, prec,
                                                                   prior1, prior2,
                                                                   level=0.95, n.max=100000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  
  
  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.hpdlimits(accepted.diff, prec,
                                                          prior1, prior2, 
                                                          level, n.max=n.max, both=T)
  }
  
  
  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, 
                     level=level, n.max=n.max, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.avg.hpdlimits.bothmarg


ss.cons.norm.knownvar.prob.hpdlimits <- function(accepted.diff, prec, 
                                                           prior1, prior2, prior1.mixture.wt=0.5, 
                                                           prob=0.5, level=0.95, n.max=100000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  
  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.hpdlimits(accepted.diff, prec,
                                                           prior1, prior2, 
                                                           prob, level=level, n.max=n.max, prior1.mixture.wt=prior1.mixture.wt)
  }


  out <- c(out, list(accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     prob=prob, level=level, n.max=n.max, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.prob.hpdlimits


ss.cons.norm.knownvar.prob.hpdlimits.mymarg <- function(accepted.diff, prec,
                                                                  prior1, prior2, clinical.prior,
                                                                  prob=0.5, level=0.95, n.max=100000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)

  
  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.hpdlimits(accepted.diff, prec,
                                                           prior1, prior2,
                                                           prob, level=level, n.max=n.max,
                                                           clinical.prior=clinical.prior)
  }
  

  out <- c(out, list(accepted.diff=accepted.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     prob=prob, level=level, n.max=n.max, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.prob.hpdlimits.mymarg


ss.cons.norm.knownvar.prob.hpdlimits.bothmarg <- function(accepted.diff, prec,
                                                                    prior1, prior2,
                                                                    prob=0.5, level=0.95, n.max=100000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)

  
  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling((abs(prior1$mu0-prior2$mu0)/accepted.diff-1)*prior1$prec0/prec)
    n <- max(n, 0)
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.hpdlimits(accepted.diff, prec,
                                                           prior1, prior2,
                                                           prob, level=level, n.max=n.max, both=T)
  }


  out <- c(out, list(accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2,
                     prob=prob, level=level, n.max=n.max, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.prob.hpdlimits.bothmarg


ss.cons.norm.knownvar.worst.cdf <- function(cdf.points, accepted.cdf.diff, prec,
                                                       prior1, prior2, 
                                                       n.start=1000, n.max=100000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)

                       
  mu0s     <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  
  worst.g <- function(n, cdf.points, prec, mu0s, prec0s)
  {
    cdf.points <- rep(cdf.points, rep(2, length(cdf.points)))
    j <- seq(from=1, to=length(cdf.points), by=2)

    if (n == 0)
    {
      p <- pnorm(cdf.points, mean=mu0s, sd=1/sqrt(prec0s))
      res <- max(abs(p[j]-p[j+1]))
    }
    else
    { 
      prec.n  <- prec0s + n*prec
      a <- sqrt(prec.n)*cdf.points - sqrt(prec.n)*prec0s*mu0s/prec.n
      b <- n*prec/sqrt(prec.n)
  
      B <- b[2]/b[1]
      A <- a[j+1] - B*a[j]
      
      if (B != 1)
      {
        delta <- (2*A*B)^2 - 4*(B*B-1)*(A*A-2*log(B))
        w <- seq(along=delta)[delta>=0]
        num1 <- -2*A*B
        num1 <- num1[w]
        denom <- 2*(B*B-1) # scalar
        delta <- delta[w]
  
        zmax <- c((num1-sqrt(delta))/denom, (num1+sqrt(delta))/denom)
      }
      else
      {
        zmax <- -A/2
        w <- seq(along=A)
      }
      
      gmax <- abs(pnorm(zmax) - pnorm(A[w] + B * zmax))
  
      res <- max(gmax)
    }
    
    return(res)
  }

  
  # ~*~*~*~*~*~

  # Initial values

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=accepted.cdf.diff, increasing.outcome.with.n=F, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    z <- worst.g(n, cdf.points, prec, mu0s, prec0s)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- .ss.nextn4nonrndoutcomes(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome),
       cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec,
       prior1=prior1, prior2=prior2, n.start=n.start, n.max=n.max, exact.results=T)
} # end of ss.cons.norm.knownvar.worst.cdf


ss.cons.norm.knownvar.avg.cdf <- function(cdf.points, accepted.cdf.diff, prec,
                                                     prior1, prior2, prior1.mixture.wt=0.5, 
                                                     n.start=1000, n.max=100000, convexity.check.sim.size=10000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)

  
  out <- .ss.cons.norm.knownvar.avg.cdf(cdf.points, accepted.cdf.diff, prec,
                                                   prior1, prior2, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt, convexity.check.sim.size=convexity.check.sim.size)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     n.start=n.start, n.max=n.max, convexity.check.sim.size=convexity.check.sim.size, exact.results=T))
                     
  if (out$convexity.check.success.rate < 1) warning("Our sample size calculations are based on a log-convexity assumption for p_nj(x.bar).\n  While we have never come across an example that violates this assumption, it appears that you may have found a counter-example.\n  Could you please send us your full problem description?")
  return(out)
} # end of ss.cons.norm.knownvar.avg.cdf


ss.cons.norm.knownvar.avg.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prec,
                                                            prior1, prior2, clinical.prior,
                                                            n.start=1000, n.max=100000, convexity.check.sim.size=10000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)
  
  out <- .ss.cons.norm.knownvar.avg.cdf(cdf.points, accepted.cdf.diff, prec,
                                                   prior1, prior2, 
                                                   n.start, n.max,
                                                   clinical.prior=clinical.prior, convexity.check.sim.size=convexity.check.sim.size)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     n.start=n.start, n.max=n.max, convexity.check.sim.size=convexity.check.sim.size, exact.results=T))

  if (out$convexity.check.success.rate < 1) warning("Our sample size calculations are based on a log-convexity assumption for  p_nj(x.bar).\n  While we have never come across an example that violates this assumption, it appears that you may have found a counter-example.\n  Could you please send us your full problem description?")
  return(out)
} # end of ss.cons.norm.knownvar.avg.cdf.mymarg


ss.cons.norm.knownvar.avg.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prec,
                                                             prior1, prior2, 
                                                             n.start=1000, n.max=100000, convexity.check.sim.size=10000)
{
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)

    
  out <- .ss.cons.norm.knownvar.avg.cdf(cdf.points, accepted.cdf.diff, prec,
                                                   prior1, prior2, n.start, n.max,
                                                   both=T, convexity.check.sim.size=convexity.check.sim.size)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec, prior1=prior1, prior2=prior2,
                     n.start=n.start, n.max=n.max, convexity.check.sim.size=convexity.check.sim.size, exact.results=T))

  if (out$convexity.check.success.rate < 1) warning("Our sample size calculations are based on a log-convexity assumption for  p_nj(x.bar).\n  While we have never come across an example that violates this assumption, it appears that you may have found a counter-example.\n  Could you please send us your full problem description?")
  return(out)
} # end of ss.cons.norm.knownvar.avg.cdf.bothmarg


ss.cons.norm.avg.hpdlimits <- function(accepted.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                 level=0.95, n.start=1000, n.max=100000, sim.size=50000, 
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  
  
  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           prior1.mixture.wt=prior1.mixture.wt)                                  
    
  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     level=level, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.hpdlimits


ss.cons.norm.avg.hpdlimits.mymarg <- function(accepted.diff, prior1, prior2, clinical.prior, 
                                                        level=0.95, n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")

  
  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           clinical.prior=clinical.prior)
    
  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     level=level, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.hpdlimits.mymarg


ss.cons.norm.avg.hpdlimits.bothmarg <- function(accepted.diff, prior1, prior2, 
                                                          level=0.95, n.start=1000, n.max=100000, sim.size=50000,
                                                          max.cons.steps.same.dir=3)
{  
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")


  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir, both=T)

  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                     level=level, n.start=n.start, n.max=n.max, sim.size=sim.size, 
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))    
  return(out)
} # end of ss.cons.norm.avg.hpdlimits.bothmarg


ss.cons.norm.prob.hpdlimits <- function(accepted.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                  prob=0.5, level=0.95, n.start=1000, n.max=100000, sim.size=50000, 
                                                  max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           prior1.mixture.wt=prior1.mixture.wt, prob, increasing.outcome.with.n=T)
    
  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     prob=prob, level=level, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.hpdlimits


ss.cons.norm.prob.hpdlimits.mymarg <- function(accepted.diff, prior1, prior2, clinical.prior, 
                                                         prob=0.5, level=0.95, n.start=1000, n.max=100000, sim.size=50000,
                                                         max.cons.steps.same.dir=3)
{  
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")

  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           clinical.prior=clinical.prior,
                                           prob=prob, increasing.outcome.with.n=T)
    
  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     prob=prob, level=level, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.hpdlimits.mymarg


ss.cons.norm.prob.hpdlimits.bothmarg <- function(accepted.diff, prior1, prior2,
                                                           prob=0.5, level=0.95, n.start=1000, n.max=100000, sim.size=50000,
                                                           max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.hpdlimits(accepted.diff, prior1, prior2,
                                           level, sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           both=T, prob=prob, increasing.outcome.with.n=T)

  out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                     prob=prob, level=level, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.hpdlimits.bothmarg


ss.cons.norm.avg.cdf <- function(cdf.points, accepted.cdf.diff, 
                                            prior1, prior2, prior1.mixture.wt=0.5, 
                                            n.start=1000, n.max=100000, sim.size=50000,
                                            max.cons.steps.same.dir=3)
{  
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2,
                                      sim.size, n.start, n.max, max.cons.steps.same.dir, 
                                      prior1.mixture.wt=prior1.mixture.wt)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.cdf


ss.cons.norm.avg.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, 
                                                  prior1, prior2, clinical.prior,
                                                  n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{  
  # prior1, prior2 and clinical.prior must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")

  
  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2, 
                                      sim.size, n.start, n.max, max.cons.steps.same.dir,
                                      clinical.prior=clinical.prior)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.cdf.mymarg


ss.cons.norm.avg.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, 
                                                     prior1, prior2, 
                                                     n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  
  
  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2,
                                      sim.size, n.start, n.max, max.cons.steps.same.dir, both=T)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2,
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.cdf.bothmarg


ss.cons.norm.prob.cdf <- function(cdf.points, accepted.cdf.diff, 
                                             prior1, prior2, prior1.mixture.wt=0.5, 
                                             prob=0.5, n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2,
                                      sim.size, n.start, n.max, max.cons.steps.same.dir, 
                                      prior1.mixture.wt=prior1.mixture.wt, prob=prob, increasing.outcome.with.n=T)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     prob=prob, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.cdf


ss.cons.norm.prob.cdf.mymarg <- function(cdf.points, accepted.cdf.diff,  
                                                    prior1, prior2, clinical.prior,
                                                    prob=0.5, n.start=1000, n.max=100000, sim.size=50000,
                                                    max.cons.steps.same.dir=3)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")
  
 
  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2,
                                      sim.size, n.start, n.max, max.cons.steps.same.dir,
                                      clinical.prior=clinical.prior,
                                      prob=prob, increasing.outcome.with.n=T)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                     prob=prob, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.cdf.mymarg


ss.cons.norm.prob.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, 
                                                      prob=0.5, n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  
  
  out <- .ss.cons.norm.cdf(cdf.points, accepted.cdf.diff, 
                                      prior1, prior2,
                                      sim.size, n.start, n.max, max.cons.steps.same.dir, 
                                      both=T, prob=prob, increasing.outcome.with.n=T)
    
  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff,  
                     prior1=prior1, prior2=prior2,
                     prob=prob, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.cdf.bothmarg


ss.cons.norm.knownvar.prob.cdf <- function(cdf.points, accepted.cdf.diff, prec,
                                                      prior1, prior2, prior1.mixture.wt=0.5, 
                                                      prob=0.5, n.start=1000, n.max=100000, epsilon=1e-8)
{
  # prior1 and prior2 must be lists with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)

  
  out <- .ss.cons.norm.knownvar.prob.cdf(cdf.points, accepted.cdf.diff, prec, prior1, prior2, n.start, n.max, prob, prior1.mixture.wt=prior1.mixture.wt, epsilon=epsilon)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec,
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt,
                     prob=prob, n.start=n.start, n.max=n.max, epsilon=epsilon, exact.results=T))
                     
  return(out)
} #end of ss.cons.norm.knownvar.prob.cdf


ss.cons.norm.knownvar.prob.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prec, 
                                                            prior1, prior2, clinical.prior, 
                                                            prob=0.5, n.start=1000, n.max=100000, epsilon=1e-8)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)
  
  
  out <- .ss.cons.norm.knownvar.prob.cdf(cdf.points, accepted.cdf.diff, prec, prior1, prior2, n.start, n.max, prob, clinical.prior=clinical.prior, epsilon=epsilon)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec,
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob,
                     n.start=n.start, n.max=n.max, epsilon=epsilon, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.prob.cdf.mymarg


ss.cons.norm.knownvar.prob.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prec,
                                                              prior1, prior2,
                                                              prob=0.5, n.start=1000, n.max=100000, epsilon=1e-8)
{
  # prior1 and prior2 must be lists with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  
  
  out <- .ss.cons.norm.knownvar.prob.cdf(cdf.points, accepted.cdf.diff, prec, prior1, prior2, n.start, n.max, prob, both=T, epsilon=epsilon)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prec=prec, prior1=prior1, prior2=prior2, 
                     prob=prob, n.start=n.start, n.max=n.max, epsilon=epsilon, exact.results=T))
  return(out)
} # end of ss.cons.norm.knownvar.prob.cdf.bothmarg


ss.cons.norm.knownvar.avg.q <- function(quantiles, accepted.diff, prec, 
                                                          prior1, prior2, prior1.mixture.wt=0.5, 
                                                          n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.q(quantiles, accepted.diff, prec,
                                                          prior1, prior2, prior1.mixture.wt=prior1.mixture.wt, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2,
                     prior1.mixture.wt=prior1.mixture.wt, n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.avg.q


ss.cons.norm.knownvar.avg.q.mymarg <- function(quantiles, accepted.diff, prec, 
                                                                 prior1, prior2, clinical.prior, 
                                                                 n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.q(quantiles, accepted.diff, prec,
                                                          prior1, prior2, clinical.prior=clinical.prior, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.avg.q.mymarg


ss.cons.norm.knownvar.avg.q.bothmarg <- function(quantiles, accepted.diff, prec, 
                                                                   prior1, prior2,  
                                                                   n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.avg.q(quantiles, accepted.diff, prec, prior1, prior2, both=T, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, prior1=prior1, prior2=prior2,
                     n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.avg.q.bothmarg


ss.cons.norm.knownvar.prob.q <- function(quantiles, accepted.diff, prec, 
                                                           prior1, prior2, prior1.mixture.wt=0.5, prob=0.5, 
                                                           n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.q(quantiles, accepted.diff, prec,
                                                           prior1, prior2, prior1.mixture.wt=prior1.mixture.wt,
                                                           prob=prob, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     prob=prob, n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.prob.q


ss.cons.norm.knownvar.prob.q.mymarg <- function(quantiles, accepted.diff, prec, 
                                                                  prior1, prior2, clinical.prior, prob=0.5, 
                                                                  n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.q(quantiles, accepted.diff, prec,
                                                           prior1, prior2, clinical.prior=clinical.prior, 
                                                           prob=prob, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob,
                     n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.prob.q.mymarg


ss.cons.norm.knownvar.prob.q.bothmarg <- function(quantiles, accepted.diff, prec, 
                                                                    prior1, prior2, prob=0.5, 
                                                                    n.max=100000)
{                                                                      
  # Both prior2 and prior1 must be lists, with dimensions mu0 and prec0
  tmp <- .ss.cons.norm.check.list(prior1, "prior1", T)
  tmp <- .ss.cons.norm.check.list(prior2, "prior2", T)


  if (prior1$prec0 == prior2$prec0)
  {
    n <- ceiling(pmax(0, (abs(prior1$mu0 - prior2$mu0)/accepted.diff - 1) * prior1$prec0/prec))
    out <- list(n=n)
  }
  else
  {
    out <- .ss.cons.norm.knownvar.prob.q(quantiles, accepted.diff, prec,
                                                           prior1, prior2, both=T, 
                                                           prob=prob, n.max=n.max)
  }
  

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prec=prec, 
                     prior1=prior1, prior2=prior2, prob=prob,
                     n.max=n.max, exact.results=T))

  return(out)
} # end of ss.cons.norm.knownvar.prob.q.bothmarg


ss.cons.norm.avg.q <- function(quantiles, accepted.diff, 
                                                 prior1, prior2, prior1.mixture.wt=0.5, 
                                                 n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir, prior1.mixture.wt=prior1.mixture.wt)


  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     n.start=n.start, n.max=n.max, sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.q


ss.cons.norm.avg.q.mymarg <- function(quantiles, accepted.diff, 
                                                        prior1, prior2, clinical.prior,
                                                        n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")
  

  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir, clinical.prior=clinical.prior)

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.q.mymarg


ss.cons.norm.avg.q.bothmarg <- function(quantiles, accepted.diff, 
                                                          prior1, prior2,
                                                          n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir, both=T)

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, 
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.avg.q.bothmarg


ss.cons.norm.prob.q <- function(quantiles, accepted.diff, 
                                                  prior1, prior2, prior1.mixture.wt=0.5, prob=0.5, 
                                                  n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")


  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           prior1.mixture.wt=prior1.mixture.wt, prob=prob, increasing.outcome.with.n=T)

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                     prob=prob, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.q


ss.cons.norm.prob.q.mymarg <- function(quantiles, accepted.diff, 
                                                         prior1, prior2, clinical.prior, prob=0.5,
                                                         n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.norm.check.list(clinical.prior, "clinical.prior")
  

  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           clinical.prior=clinical.prior, prob=prob, increasing.outcome.with.n=T)

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff,
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob,
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.q.mymarg


ss.cons.norm.prob.q.bothmarg <- function(quantiles, accepted.diff, prior1, prior2,
                                                           prob=0.5, n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.norm.check.list(prior2, "prior2")


  out <- .ss.cons.norm.q(quantiles, accepted.diff, prior1, prior2,
                                           sim.size, n.start, n.max, max.cons.steps.same.dir,
                                           both=T, prob=prob, increasing.outcome.with.n=T)

  out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff,
                     prior1=prior1, prior2=prior2, prob=prob,
                     n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  return(out)
} # end of ss.cons.norm.prob.q.bothmarg


ss.plot <- function(ss.out, zero=0.1, show.fit=T)
{
  log.n <- log(pmax(zero, ss.out$n.visited))
  n.plot <- pmax(exp(zero), ss.out$n.visited)
  out <- ss.out$outcome
  plot(n.plot, out, xlab='Sample Size', ylab='Outcome', xlog=T)

  if (show.fit)
  {
    df <- data.frame(out=out, log.n=log.n, log.n2 = log.n^2)
    glm.out <- glm(out~log.n+log.n2, data=df)
    log.n <- log(pmax(zero, par('usr')[c(1,2)]))
    log.n <- seq(from=log.n[1], to=log.n[2], length=300)
    df <- data.frame(log.n=log.n, log.n2=log.n^2)
    predict.out <- predict(glm.out, df)
    points(exp(log.n), predict.out, type='l', col='orange')
  }
} # end of ss.plot


# ----------------------------------------------------------------------------------
# --- Definition of internal functions, not be called directly by user -------------
#     (will be called by other functions with proper arguments)        -------------


.ss.cons.norm.hpdlimits <- function(accepted.diff, prior1, prior2,
                                              level, sim.size, n.start, n.max, mcs, min.for.possible.return=2^ceiling(1.5*mcs), next.n=.ss.bsearch,
                                              clinical.prior=list(),
                                              prior1.mixture.wt=1, both=F, prob=numeric(0), increasing.outcome.with.n=F)
{
  return.prob <- length(prob) > 0
  
  mu0s <- c(prior1$mu0, prior2$mu0)
  n0s <- c(prior1$n0, prior2$n0)
  alphas <- c(prior1$prec.shape, prior2$prec.shape)
  betas  <- c(prior1$prec.rate,  prior2$prec.rate)
  mu03 <- clinical.prior$mu0
  alpha3 <- clinical.prior$prec.shape
  beta3  <- clinical.prior$prec.rate
  n03    <- clinical.prior$n0
  

  h <- function(n, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, accepted.diff, return.prob, level)
  {
    if (n == 0)
    {
      margins <- sqrt(betas/n0s/alphas) * qt((1+level)/2, df=2*alphas)
      res <- abs(diff(mu0s)) + abs(diff(margins))
      
      if (return.prob)
      {
        res <- as.double(res<=accepted.diff)
      }
    }
    else
    {
      if (both)
      {
        w <- seq(2)
        sim.size <- 2 * sim.size
      }
      else if (length(alpha3) == 2)
      {
        w <- sample(c(1,2), size=sim.size, replace=T, prob=c(prior1.mixture.wt,1-prior1.mixture.wt))
      }
      else
      {
        w <- 1
      }

      rnd.post.moments <- .ss.cons.norm.PostMoments.sample(n, sim.size, w, alpha3, beta3, mu03, n03, alphas, betas, mu0s, n0s)
      
      hpd.diff <- abs(rnd.post.moments$pm1$mu.n - rnd.post.moments$pm2$mu.n) + abs(qt((1+level)/2,rnd.post.moments$pm1$alpha.n)/sqrt(rnd.post.moments$pm1$lambda.n) - qt((1+level)/2,rnd.post.moments$pm2$alpha.n)/sqrt(rnd.post.moments$pm2$lambda.n))
      hpd.diff <- matrix(hpd.diff, ncol=both+1, byrow=T) 
      
      if (return.prob) hpd.diff <- (hpd.diff <= accepted.diff)
      hpd.diff <- (matrix(1, nrow=1, ncol=nrow(hpd.diff)) %*% hpd.diff) / nrow(hpd.diff) # means

      res <- ifelse(both, ifelse(return.prob, min(hpd.diff), max(hpd.diff)), hpd.diff)
    }
    
    return(res)
  } # end of h

  # --- Initial values ---
  
  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  
  target <- ifelse(return.prob, prob, accepted.diff)
  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, nmax=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)

  if (length(mu03) == 0)
  {
    mu03 <- mu0s
    n03 <- n0s
    alpha3 <- alphas
    beta3 <- betas
  }

  while (out$continue)
  {
    z <- h(n, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, accepted.diff, return.prob, level)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.hpdlimits


.ss.cons.norm.q <- function(quantiles, accepted.diff, prior1, prior2,
                                              sim.size, n.start, n.max, mcs, min.for.possible.return=2^ceiling(1.5*mcs), next.n=.ss.bsearch,
                                              clinical.prior=list(),
                                              prior1.mixture.wt=1, both=F, prob=numeric(0), increasing.outcome.with.n=F)
{
  return.prob <- length(prob) > 0
  
  mu0s <- c(prior1$mu0, prior2$mu0)
  n0s <- c(prior1$n0, prior2$n0)
  alphas <- c(prior1$prec.shape, prior2$prec.shape)
  betas  <- c(prior1$prec.rate,  prior2$prec.rate)
  mu03 <- clinical.prior$mu0
  alpha3 <- clinical.prior$prec.shape
  beta3  <- clinical.prior$prec.rate
  n03    <- clinical.prior$n0
  

  qdiff <- function(n, quantiles, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, accepted.diff, return.prob)
  {
    q.len <- length(quantiles)
    
    if (n == 0)
    {
      t.df <- 2*alphas
      lambdas <- betas/n0s/alphas
      q <- rep(quantiles, rep(2, q.len))
      t.stats <- qt(q, t.df)
      q <- mu0s + t.stats/sqrt(lambdas)
      q <- matrix(q, nrow=2)
      qdiff <- abs(q[1,]-q[2,])
      
      res <- max(qdiff)
      if (return.prob) res <- as.double(res<=accepted.diff)
    }
    else
    {
      if (both)
      {
        w <- seq(2)
        sim.size <- 2 * sim.size
      }
      else if (length(alpha3) == 2)
      {
        w <- sample(c(1,2), size=sim.size, replace=T, prob=c(prior1.mixture.wt,1-prior1.mixture.wt))
      }
      else
      {
        w <- 1
      }


      rnd.post.moments <- .ss.cons.norm.PostMoments.sample(n, sim.size, w, alpha3, beta3, mu03, n03, alphas, betas, mu0s, n0s)
      
      t1.stats <- qt(quantiles, rnd.post.moments$pm1$alpha.n)
      t2.stats <- qt(quantiles, rnd.post.moments$pm2$alpha.n)
      which.t <- rep(seq(q.len), rep(length(rnd.post.moments$pm1$mu.n), q.len)) 
      qdiff <- rnd.post.moments$pm1$mu.n - rnd.post.moments$pm2$mu.n + t1.stats[which.t]/sqrt(rnd.post.moments$pm1$lambda.n) - t2.stats[which.t]/sqrt(rnd.post.moments$pm2$lambda.n)
      qdiff <- matrix(qdiff, ncol=q.len)
      qdiff <- apply(abs(qdiff), 1, max)
      
      if (return.prob) qdiff <- (qdiff <= accepted.diff)
      ncol <- ifelse(both, 2, 1)
      qdiff <- matrix(qdiff, byrow=T, ncol=ncol)
      qdiff <- (matrix(1, nrow=1, ncol=nrow(qdiff)) %*% qdiff) / nrow(qdiff) # means

      res <- ifelse(both, ifelse(return.prob, min(qdiff), max(qdiff)), qdiff)
    }
    
    return(res)
  } # end of qdiff

  # --- Initial values ---
  
  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  
  target <- ifelse(return.prob, prob, accepted.diff)
  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, nmax=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)

  if (length(mu03) == 0)
  {
    mu03 <- mu0s
    n03 <- n0s
    alpha3 <- alphas
    beta3 <- betas
  }

  while (out$continue)
  {
    z <- qdiff(n, quantiles, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, accepted.diff, return.prob)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.q


.ss.cons.norm.cdf <- function(cdf.points, accepted.cdf.diff,
                                         prior1, prior2, 
                                         sim.size, n.start, n.max, mcs, min.for.possible.return=2^ceiling(1.5*mcs), next.n=.ss.bsearch, 
                                         clinical.prior=list(), 
                                         prior1.mixture.wt=1, both=F, prob=numeric(0), increasing.outcome.with.n=F)
{
  return.prob <- length(prob) > 0

  mu0s <- c(prior1$mu0, prior2$mu0)
  n0s <- c(prior1$n0, prior2$n0)
  alphas <- c(prior1$prec.shape, prior2$prec.shape)
  betas  <- c(prior1$prec.rate,  prior2$prec.rate)
  mu03   <- clinical.prior$mu0
  alpha3 <- clinical.prior$prec.shape
  beta3  <- clinical.prior$prec.rate
  n03    <- clinical.prior$n0

  
  g <- function(n, cdf.points, accepted.cdf.diff, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, return.prob)
  {
    if (n == 0)
    {
      marg.mu <- mu0s
      marg.prec <- n0s*alphas/betas
      marg.alpha <- 2*alphas

      cdf.points <- rep(cdf.points, rep(2, length(cdf.points)))
      pdiff <- pt((cdf.points-marg.mu)*sqrt(marg.prec), marg.alpha)
      j <- seq(from=1, to=length(pdiff), by=2)
      pdiff <- abs(pdiff[j] - pdiff[j+1])
      res <- max(pdiff)

      if (return.prob)
      {
        res <- as.double(res<=accepted.cdf.diff)
      }
    }
    else
    {
      if (both)
      {
        w <- seq(2)
        sim.size <- 2 * sim.size
      }
      else if (length(alpha3) == 2)
      {
        w <- sample(c(1,2), size=sim.size, replace=T, prob=c(prior1.mixture.wt,1-prior1.mixture.wt))
      }
      else
      {
        w <- 1
      }

      rnd.post.moments <- .ss.cons.norm.PostMoments.sample(n, sim.size, w, alpha3, beta3, mu03, n03, alphas, betas, mu0s, n0s)

      Q <- length(cdf.points)
      w <- rep(seq(sim.size), rep(Q, sim.size))
      pdiff <- abs(pt((cdf.points-rnd.post.moments$pm1$mu.n[w])*sqrt(rnd.post.moments$pm1$lambda.n[w]), rnd.post.moments$pm1$alpha.n) - pt((cdf.points-rnd.post.moments$pm2$mu.n[w])*sqrt(rnd.post.moments$pm2$lambda.n[w]), rnd.post.moments$pm2$alpha.n))

      if (Q > 1)
      {
        pdiff <- matrix(pdiff, nrow=Q)
        pdiff <- apply(pdiff, 2, max)
      }

      pdiff <- matrix(pdiff, ncol=both+1, byrow=T)

      if (return.prob) pdiff <- (pdiff <= accepted.cdf.diff)
      pdiff <- (matrix(1, nrow=1, ncol=nrow(pdiff)) %*% pdiff) / nrow(pdiff) # means

      res <- ifelse(both, ifelse(return.prob, min(pdiff), max(pdiff)), pdiff)
    }

    return(res)
  } # end of g

  if (length(mu03) == 0)
  {
    mu03   <- mu0s
    n03    <- n0s
    alpha3 <- alphas
    beta3  <- betas
  }


  # --- Initial values ---
  
  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  target <- ifelse(return.prob, prob, accepted.cdf.diff)
  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, nmax=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)

  while (out$continue)
  {
    z <- g(n, cdf.points, accepted.cdf.diff, mu0s, n0s, alphas, betas, mu03, n03, alpha3, beta3, prior1.mixture.wt, both, sim.size, return.prob)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.cdf


.ss.cons.norm.knownvar.avg.cdf <- function(cdf.points, accepted.cdf.diff, prec,
                                                      prior1, prior2, 
                                                      n.start, n.max, next.n=.ss.nextn4nonrndoutcomes,
                                                      clinical.prior=list(), 
                                                      prior1.mixture.wt=1, both=F, convexity.check.sim.size=10000)
{                    
  mu0s   <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  

  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
  }
  else
  {
    nu  <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }
   

  if (is.na(match("mvtnorm", rownames(installed.packages()))))
  {
    options(CRAN="http://cran.us.r-project.org")
    install.packages("mvtnorm")
  }
  if (is.na(match("mvtnorm", (.packages())))) library(mvtnorm)
  
  
  # ... Useful functions .......................................................
  
  
  ab.parms <- function(n, z, prec, mu0s, prec0s, tau)
  {
    z <- sort(z)
    sqrt.prec.n  <- sqrt(prec0s + n*prec)
    
    a.n <- matrix(NA, nrow=length(z), ncol=2)
    a.n <- sqrt.prec.n[col(a.n)]*z[row(a.n)] - prec0s[col(a.n)]*mu0s[col(a.n)]/sqrt.prec.n[col(a.n)]
    a.n <- matrix(a.n, ncol=2)
      
    b.n <- -n*prec/sqrt.prec.n
    tau.n <- n*tau*prec/(tau+n*prec) # x.bar ~ N(nu, tau.n) 
    
    list(sqrt.prec.n=sqrt.prec.n, a.n=a.n, b.n=b.n, tau.n=tau.n)
  } # end of ab.parms
  
  
  DiamondBite <- function(a, b, k, nu, tau.n)
  {
    # a, b, k   : numeric(2)
    # nu, tau.n : numeric(1)

    x.intersect <- -diff(a)/diff(b)
    k <- sort(k)
    left <- k[1] < x.intersect
    k.LargeBite.side <- ifelse(left, 1, 2)

    area <- TinyBite(a, b, k[k.LargeBite.side], nu, tau.n) + 
            LargeBite(a, b, k[3-k.LargeBite.side], nu, tau.n) -
            SmallSlice(a, b, left, nu, tau.n)

    area  
  } # end of DiamondBite
  

  g.mean <- function(n, cdf.points, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
  {
    need.convexity.check <- F
    
    if (n == 0)
    {
      res <- pnorm(cdf.points, mean=mu0s[1], sd=1/sqrt(prec0s[1])) - pnorm(cdf.points, mean=mu0s[2], sd=1/sqrt(prec0s[2]))
      res <- max(abs(res))
    }
    else 
    {      
      cdf.points <- sort(cdf.points)
      
      tmp <- ab.parms(n, cdf.points, prec, mu0s, prec0s, tau)
      sqrt.prec.n <- tmp$sqrt.prec.n
      a.n <- tmp$a.n 
      b.n <- tmp$b.n
      tau.n <- tmp$tau.n
      
      
      if (length(cdf.points) == 1)
      {
        if (length(unique(prec0s)) == 1)
        {
          b.n <- b.n[1] # b.n[1] == b.n[2]
          z.mean <- -b.n*nu
          z.var <- 1 + b.n^2/tau.n

          w <- seq(2)
          if (length(nu) == 2) w <- rep(w, c(2,2))
          p <- pnorm(a.n[w], mean=z.mean, sd=sqrt(z.var))
          p <- matrix(p, ncol=2)
          p <- p[,2] - p[,1]

          if (length(nu) == 2)
          {
            res <- ifelse(both, min(p), sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*p))
          }
          else
          {
            res <- p
          }
        }
        else
        {   
          x.star <- -diff(a.n)/diff(b.n) # intersection of the two lines a.n + b.n*xbar
          z.star <- a.n[1] + b.n[1]*x.star

          p <- numeric(length(nu))
          for (i in seq(along=p))
          {
            p[i] <- ratio.of.two.independent.normal.distrns.cdf(b.n, -z.star, nu[i]-x.star, 1, tau.n[i])
          }  

          if (length(nu) == 2)
          {
            res <- ifelse(both, min(p), sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*p))
          }
          else
          {
            res <- p
          }
        }
      }
      else if (length(unique(prec0s)) == 1)
      {
        # lambda_01 == lambda_02

        mu0s <- sort(mu0s)
        prec0 <- unique(prec0s)
        sqrt.prec.n <- sqrt(prec0 + n*prec)

        x.opt <- cdf.points + prec0/(n*prec)*(cdf.points-mean(mu0s))
        m <- length(x.opt)
        x.switch <- matrix(0, nrow=m-1, ncol=m)
        diag(x.switch) <- 1
        x.switch[col(x.switch) == row(x.switch)+1] <- 1
        x.switch <- as.vector((x.switch%*%x.opt)/2)

        nu.len <- length(nu)
        res <- numeric(nu.len)        

        for (nu.index in seq(along=nu))
        {
          # left-side open area
          # + right-side open area
          res[nu.index] <- LargeBite(a.n[1,], b.n, x.switch[1], nu[nu.index], tau[nu.index]) +
                           LargeBite(a.n[nrow(a.n),], b.n, rev(x.switch)[1], nu[nu.index], tau[nu.index])

          # and the remaining close areas
          j <- seq(along=x.switch)
          j <- j[-length(j)]

          for (i in j)
          {
            k <- c(i, i+1)
            res[nu.index] <- res[nu.index] + DiamondBite(a.n[k,], b.n, x.switch[k], nu[nu.index], tau[nu.index])
          }
        }

        if (nu.len > 1)
        {
          if (both)
          {
            res <- max(res)
          }
          else
          {
            res <- sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*res)
          }
        }
      }
      else
      {
        res <- TotArea(a.n, b.n, sqrt.prec.n, nu, tau.n, prior1.mixture.wt, both)
        need.convexity.check <- T
      }
    }
    
    out <- list(res=res, need.convexity.check=need.convexity.check)
  } # end of g.mean


  LargeBite <- function(a, b, k, nu, tau.n)
  {
    # a, b      : numeric(2)
    # k         : numeric(1)
    # nu, tau.n : numeric(1)

    x.intersect <- -diff(a)/diff(b)
    left <- k < x.intersect
    sd <- 1/sqrt(tau.n)
    p.left <- pnorm(k, mean=nu, sd=sd)
    which.b.upper <- ifelse(left, 1, 2)
    which.b.lower <- 3 - which.b.upper
    area <- ifelse(left, p.left, 1-p.left) - 
            LargeSlice(a[which.b.upper], b[which.b.upper], k, T, left, nu, tau.n) - 
            LargeSlice(a[which.b.lower], b[which.b.lower], k, F, left, nu, tau.n)

    area 
  } # end of LargeBite


  LargeSlice <- function(a, b, k, upper, left, nu, tau.n)
  {
    # a, b, k      : numeric(1)
    # upper, left  : logical(1)
    # nu, tau.n    : numeric(1)

    y.star <- a + b*k

    mu <- -a - b*nu ; sd <- sqrt(1 + b^2/tau.n)
    above.area <- pnorm(0, mean=mu, sd=sd, lower.tail=F)
    sd <- 1/sqrt(tau.n)
    right.area <- pnorm(k, mean=nu, sd=sd, lower.tail=F) 
    opposite.slices <- ratio.of.two.independent.normal.distrns.cdf(b, -y.star, nu-k, 1, tau.n)
    areas <- c(1, above.area, right.area, opposite.slices)

    w <- numeric(4)
    w[1] <- ifelse(!upper & left, 2, 0)
    w[2] <- ifelse(upper, 1, -1)
    w[3] <- ifelse(left, -1, 1)
    w[4] <- ifelse(xor(upper, !left), 1, -1)

    sum(w*areas)/2
  } # end of LargeSlice


  log.diff <- function(a, b)
  {
    # a, b: numeric(same length)

    l <- matrix(c(a,b), ncol=2)
    J <- matrix(c(1,-1), nrow=2, ncol=1)
    m <- pmax(a, b)
    x <- as.vector(exp(l-m)%*%J)
    s <- sign(x)
    x <- log(abs(x)) + m
    list(sign=s, log=x)
  } # end of log.diff


  mixed.NewtonRaphson.bisectional <- function(an.matrix, bn.matrix, x.lim, left2right=T, x.start=numeric(0), epsilon=1e-8)
  {
    if (length(x.start) == 0)
    {
      if (any(is.infinite(x.lim)))
      {
        w.infinite <- which(is.infinite(x.lim))
        s <- ifelse(w.infinite==2, 1, -1)
        x.start <- x.lim[3-w.infinite] + 3*s*epsilon
      }
      else
      {
        x.start <- mean(x.lim)
      }
    }


    continue <- T
    x.new <- x.start
    pos.side <- ifelse(left2right, 1, 2)

    an <- c(t(an.matrix))
    bn <- c(t(bn.matrix))

    while (continue)
    {
      x <- x.new
      u <- an + bn*x

      f1 <- pnorm.diff.log(u[1], u[2])
      f2 <- pnorm.diff.log(u[3], u[4])
      f <- log.diff(f1, f2)

      fp <- dnorm(u, log=T) + log(abs(bn))
      fp1 <- log.diff(fp[1], fp[2])
      fp2 <- log.diff(fp[3], fp[4])
      fp <- log.diff(fp1$log, fp2$log)
      fp$sign <- -fp$sign # because bn's are negative
      x.new <- x - f$sign*fp$sign*exp(f$log-fp$log)

      # Take x as a new lower/upper limit
      side <- ifelse(f$sign==1, pos.side, 3-pos.side)
      x.lim[side] <- x

      if (x.new < min(x.lim) | x.new > max(x.lim)) x.new <- mean(x.lim)
      continue <- abs(x-x.new) >= epsilon
    }

    x.new
  } # end of mixed.NewtonRaphson.bisectional


  next.intersection <- function(j, j.section, a.n, b.n, x.cutoffs, current.x, left2right, stop.lim)
  {
    jp.list <- seq(nrow(a.n))[-j]
    jp.list <- jp.list[order(abs(jp.list-j))]
    far.end <- ifelse(left2right, 2, 1)

    x.lim <- c(-Inf, x.cutoffs[j,], Inf)
    x.lim <- x.lim[j.section+c(0,1)]

    if (left2right)
    {
      if (current.x > x.lim[1]) x.lim[1] <- current.x
    }
    else
    {
      if (current.x < x.lim[2]) x.lim[2] <- current.x
    }

    soln <- list(j=numeric(0), x=numeric(0))

    for (jp in jp.list)
    {
      k <- c(j, jp)
      tmp <- next.intersection.2curves(j.section, a.n[k,], b.n, x.cutoffs[jp,], x.lim, left2right, stop.lim)

      if (!is.na(tmp))
      {
        soln <- list(j=jp, x=tmp)
        x.lim[far.end] <- tmp
      }
    }

    soln # is a null list or a list of length 1 in both j & x
  } # end of next.intersection


  next.intersection.2curves <- function(j.section, an.matrix, b.n, jp.cutoffs, x.lim, left2right, stop.lim, epsilon=1e-8)
  {
    bn.matrix <- matrix(b.n, nrow=nrow(an.matrix), ncol=2, byrow=T)

    # Prepare a list of jp.section's to visit

    jpsection2visit <- logical(4)
    jpsection2visit[1] <- jp.cutoffs[1] > x.lim[1]
    jpsection2visit[2] <- jp.cutoffs[1] < x.lim[2] & jp.cutoffs[2] > x.lim[1]
    jpsection2visit[3] <- jp.cutoffs[2] < x.lim[2] & jp.cutoffs[3] > x.lim[1]
    jpsection2visit[4] <- jp.cutoffs[3] < x.lim[2]
    jp.cutoffs <- c(-Inf, jp.cutoffs, Inf)

    jpsection.list <- which(jpsection2visit)
    if (!left2right) jpsection.list <- rev(jpsection.list)

    intersection.point <- NA
    found.intersection <- F
    continue <- T


    while (continue)
    {
      jp.section <- jpsection.list[1]
      jpsection.list <- jpsection.list[-1]

      tmp.an <- an.matrix
      tmp.bn <- bn.matrix

      if (j.section > 2)
      { 
        tmp.an[1,] <- rev(tmp.an[1,])
        tmp.bn[1,] <- rev(tmp.bn[1,])
      }

      if (jp.section > 2)
      { 
        tmp.an[2,] <- rev(tmp.an[2,])
        tmp.bn[2,] <- rev(tmp.bn[2,])
      }

      tmp.interval <- c(max(x.lim[1], jp.cutoffs[jp.section]), min(x.lim[2], jp.cutoffs[jp.section+1]))
      far.end <- ifelse(left2right, 2, 1)
      infinite.lim <- is.infinite(tmp.interval)
      if (any(infinite.lim))
      {
        w <- which(infinite.lim)
        tmp.interval[w] <- stop.lim[w]
      }
      
      # Compute p value at far end of interval
      u <- tmp.an + tmp.bn*tmp.interval[far.end]
      pdiff <- pnorm.diff.log(u[,1], u[,2])

      if (j.section%%2 == jp.section%%2)
      {
        if (diff(pdiff) > 0)
        {
          x.start <- numeric(0)

          # Start at far.end if far.end is a zero
          if (is.infinite(pdiff[1]))
          {
            if (left2right)
            {
              x.start <- max(mean(tmp.interval), tmp.interval[far.end]-epsilon)
            }
            else
            {
              x.start <- min(mean(tmp.interval), tmp.interval[far.end]+epsilon)
            }  
          }

          intersection.point <- mixed.NewtonRaphson.bisectional(tmp.an, tmp.bn, tmp.interval, x.start=x.start, left2right=left2right)
          found.intersection <- T
        }
        else
        {
          intersection.point <- next.intersection.in.same.parity.sections(tmp.an, tmp.bn, tmp.interval, jp.section, left2right=left2right)
          found.intersection <- !is.na(intersection.point)
        }   
      }
      else if (diff(pdiff) > 0)
      {
        intersection.point <- mixed.NewtonRaphson.bisectional(tmp.an, tmp.bn, tmp.interval, left2right=left2right)
        found.intersection <- T
      }

      continue <- !found.intersection & length(jpsection.list) > 0
    }

    intersection.point
  } # end of next.intersection.2curves


  next.intersection.in.same.parity.sections <- function(an.matrix, bn.matrix, x.lim, jp.section, left2right=T)
  {
    # an.matrix, bn.matrix : 2 x 2 matrices
    # x.lim                : vector of length 2

    a2 <- an.matrix[2,]
    b2 <- bn.matrix[2,]

    # start search of potential sub-intervals with an intersection

    intervals2check <- matrix(x.lim, ncol=2)
    soln <- NA
    continue <- T

    while (continue)
    {
      current.interval <- intervals2check[1,]
      intervals2check  <- intervals2check[-1,,drop=F]

      # Compute slopes of lower curve (the second) at both ends

      u1 <- a2[1] + b2[1]*current.interval
      u2 <- a2[2] + b2[2]*current.interval

      pdiff <- pnorm.diff.log(u1, u2)

      # Compute slopes of tangents at both ends

      num <- dnorm(c(u1, u2), log=T)
      num <- matrix(num, ncol=2)
      num <- num + log(abs(b2[col(num)]))
      num <- log.diff(num[,1], num[,2])
      slopes <- -num$sign * exp(num$log - pdiff)

      # Correct infinite slopes
      # (which could happen if section is so short that its border is virtually at the zero)
      if (all(is.infinite(slopes)))
      {
        slope2correct <- ifelse(jp.section==2, 1, 2)
        slopes[slope2correct] <- 0
      }


      if (is.infinite(slopes[1]))
      {
        p2.intersect.leftSide <- pdiff[2] - diff(current.interval)*slopes[2]
        # compute p1 on the left limit of current interval
        u1 <- an.matrix[1,1] + bn.matrix[1,1]*current.interval[1]
        u2 <- an.matrix[1,2] + bn.matrix[1,2]*current.interval[1]
        p1.leftSide <- pnorm.diff.log(u1, u2)

        if (p2.intersect.leftSide > p1.leftSide)
        {
          x.middle <- mean(current.interval)
          u1 <- an.matrix[,1] + bn.matrix[,1]*x.middle
          u2 <- an.matrix[,2] + bn.matrix[,2]*x.middle
          p.middle <- pnorm.diff.log(u1, u2)

          if (diff(p.middle) > 0)
          {
            # we have found a point where 2nd curve is above 1st curve 
            # => there IS an intersection

            x.lim <- sort(c(current.interval, x.middle))
            j2drop <- ifelse(left2right, 3, 1)

            x.lim <- x.lim[-j2drop]
            soln <- mixed.NewtonRaphson.bisectional(an.matrix, bn.matrix, x.lim, left2right=left2right, x.start=x.middle)

            continue <- F
          }
          else
          {
            tmp <- matrix(sort(c(current.interval, rep(x.middle,2))), ncol=2) 
            intervals2check <- rbind(intervals2check, tmp)
          }
        }
        else
        {
          continue <- nrow(intervals2check) > 0
        }
      }
      else if (is.infinite(slopes[2]))
      {
        p2.intersect.rightSide <- pdiff[1] + diff(current.interval)*slopes[1]
        # compute p1 on the right limit of current interval
        u1 <- an.matrix[1,1] + bn.matrix[1,1]*current.interval[2]
        u2 <- an.matrix[1,2] + bn.matrix[1,2]*current.interval[2]
        p1.rightSide <- pnorm.diff.log(u1, u2)

        if (p2.intersect.rightSide > p1.rightSide)
        {
          x.middle <- mean(current.interval)
          u1 <- an.matrix[,1] + bn.matrix[,1]*x.middle
          u2 <- an.matrix[,2] + bn.matrix[,2]*x.middle
          p.middle <- pnorm.diff.log(u1, u2)

          if (diff(p.middle) > 0)
          {
            # we have found a point where 2nd curve is above 1st curve 
            # => there IS an intersection

            x.lim <- sort(c(current.interval, x.middle))
            j2drop <- ifelse(left2right, 3, 1)

            x.lim <- x.lim[-j2drop]
            soln <- mixed.NewtonRaphson.bisectional(an.matrix, bn.matrix, x.lim, left2right=left2right, x.start=x.middle)

            continue <- F
          }
          else
          {
            tmp <- matrix(sort(c(current.interval, rep(x.middle,2))), ncol=2) 
            intervals2check <- rbind(intervals2check, tmp)
          }
        }
        else
        {
          continue <- nrow(intervals2check) > 0
        }
      }
      else
      {
        # Find coordinates of the intersection of the two tangents

        diff.slopes <- diff(slopes)

        if (diff.slopes == 0)
        {
          soln <- ifelse(left2right, max(current.interval), min(current.interval))
          continue <- F
        }
        else
        {
          x.star <- (-diff(pdiff) + diff(slopes*current.interval))/diff(slopes)
          p2.intersect <- slopes[2]*(x.star-current.interval[2]) + pdiff[2]

          # Compute the height of the two curves at intersection point x.star

          u1 <- an.matrix[,1] + bn.matrix[,1]*x.star
          u2 <- an.matrix[,2] + bn.matrix[,2]*x.star
          p.star <- pnorm.diff.log(u1, u2)

          if (diff(p.star) > 0)
          {
            # we have found a point where 2nd curve is above 1st curve 
            # => there IS an intersection

            x.lim <- sort(c(current.interval, x.star))
            j2drop <- ifelse(left2right, 3, 1)

            x.lim <- x.lim[-j2drop]
            soln <- mixed.NewtonRaphson.bisectional(an.matrix, bn.matrix, x.lim, left2right=left2right, x.start=x.star)

            continue <- F
          }
          else if (p2.intersect > p.star[1])
          {
            tmp <- matrix(sort(c(current.interval, rep(x.star,2))), ncol=2) 
            intervals2check <- rbind(intervals2check, tmp)
          }
          else
          {
            continue <- nrow(intervals2check) > 0
          } 
        }
      }
    }

    soln
  } # end of next.intersection.in.same.parity.sections


  pnorm.diff.log <- function(u1, u2)
  {
    my.pnorm <- function(u)
    {
      lo <- u < 0
      which.lo <- which(lo)
      which.hi <- which(!lo)

      p <- rep(NA, length(u1))
      if (length(which.lo) > 0) p[which.lo] <- pnorm(u[which.lo], log.p=T, lower.tail=T)
      if (length(which.hi) > 0) p[which.hi] <- pnorm(u[which.hi], log.p=T, lower.tail=F)

      res <- list(p=p, lower.tail=lo)
      res
    } # end of my.pnorm


    u <- matrix(c(u1,u2), ncol=2)
    u1 <- pmax(u[,1], u[,2])
    u2 <- pmin(u[,1], u[,2])

    p1 <- my.pnorm(u1)
    p2 <- my.pnorm(u2)

    p <- rep(NA, length(u1))

    w <- which(p1$lower.tail & p2$lower.tail)
    if (length(w) > 0) p[w] <- log.diff(p1$p[w], p2$p[w])$log

    w <- which(!p1$lower.tail & !p2$lower.tail)
    if (length(w) > 0) p[w] <- log.diff(p2$p[w], p1$p[w])$log

    w <- which(!p1$lower.tail & p2$lower.tail)
    if (length(w) > 0) p[w] <- log(1 - exp(p1$p[w]) - exp(p2$p[w]))  

    p
  } # end of pnorm.diff.log

   
  ratio.of.two.independent.normal.distrns.cdf <- function(b, mu.num, mu.denom, prec.num, prec.denom)
  {
    # b: vector of length 1 or 2

    sigma.num   <- 1/sqrt(prec.num)
    sigma.denom <- 1/sqrt(prec.denom)
    ratio.of.two.correlated.normal.distrns.cdf(b, mu.num, mu.denom, sigma.num, sigma.denom, rho=0)
  } # end of ratio.of.two.independent.normal.distrns.cdf


  ratio.of.two.correlated.normal.distrns.cdf <- function(w, mu.num, mu.denom, sigma.num, sigma.denom, rho)
  {
    # Function (very) slightly modified from
    # "The R-code for computing the cdf and the df of the ratio of two correlated Normal rv"
    # De Capitani and Pollastri

    # w: can be of length 2  -> return  P(w[1] < Y/X < w[2])
    #                  or 1  -> return  P(Y/X < w)

    # definition of the quantities in expression (2)
    b <- mu.denom/sigma.denom
    a   <- sqrt(1/(1-rho^2)) * (mu.num/sigma.num-rho*b)
    t.w <- sqrt(1/(1-rho^2)) * (sigma.denom/sigma.num*w-rho)

    # introduction of auxialiry variables
    A <- (a-b*t.w)/sqrt(1+t.w^2)
    B <- -b
    C <- t.w/sqrt(1+t.w^2)

    #definition of the two addends in expression (1)

    res <- numeric(0)
    for (i in seq(along=w))
    {
      Mcor <- matrix(c(1,C[i],C[i],1),2,2)
      A1 <- pmvnorm(lower=c(A[i],B),upper=Inf, corr=Mcor)
      A2 <- pmvnorm(lower=c(-A[i],-B),upper=Inf, corr=Mcor)
      attributes(A1) <- NULL; attributes(A2) <- NULL
      res[i] <- A1 + A2
    }
  
    if (length(res) == 2) res <- abs(diff(res))  
    return(res)
  } # end of ratio.of.two.correlated.normal.distrns.cdf


  residual.prob.max <- function(a.n, b.n, x, left, nu, tau.n)
  {
    # a.n   : numeric matrix #z x 2
    # b.n   : numeric(2)
    # x     : numeric(1)
    # left  : logical(1)
    # nu    : numeric(1/2)
    # tau.n : numeric(length(nu))

    a.sup <- matrix(NA, nrow=2, ncol=2)
    a.sup[1,] <- apply(a.n, 2, min)
    a.sup[2,] <- apply(a.n, 2, max)

    x.star <- sort((a.sup[,1] - a.sup[,2])/diff(b.n))
    h.star <- sort(rev(a.sup[,1]) + b.n[1]*x.star)

    nu.len <- length(nu)
    nu.indices <- seq(nu.len)
    area <- numeric(nu.len)

    if (left)
    {
      # left-side prob
      a.LargeBite <- a.sup[col(a.sup)!=row(a.sup)]

      if (x < x.star[1])
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x, nu[nu.index], tau.n[nu.index])
        }
      }
      else if (x < x.star[2])
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x.star[1], nu[nu.index], tau.n[nu.index]) +
                            diff(pnorm(c(x.star[1], x), mean=nu[nu.index], sd=1/sqrt(tau.n[nu.index]))) * diff(pnorm(h.star))
        }
      }
      else
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x.star[1], nu[nu.index], tau.n[nu.index]) +
            diff(pnorm(x.star, mean=nu[nu.index], sd=1/sqrt(tau.n[nu.index]))) * diff(pnorm(h.star)) +
            DiamondBite(diag(a.sup), b.n, c(x.star[2], x), nu[nu.index], tau.n[nu.index])
        }
      }
    }
    else
    {
      # right-side prob
      a.LargeBite <- diag(a.sup)

      if (x < x.star[1])
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x.star[2], nu[nu.index], tau.n[nu.index]) +
                            diff(pnorm(x.star, mean=nu[nu.index], sd=1/sqrt(tau.n[nu.index]))) * diff(pnorm(h.star)) +
                            DiamondBite(a.sup[col(a.sup)!=row(a.sup)], b.n, c(x, x.star[1]), nu[nu.index], tau.n[nu.index])
        }
      }
      else if (x < x.star[2])
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x.star[2], nu[nu.index], tau.n[nu.index]) +
                            diff(pnorm(c(x, x.star[2]), mean=nu[nu.index], sd=1/sqrt(tau.n[nu.index]))) * diff(pnorm(h.star))
        }
      }
      else
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- LargeBite(a.LargeBite, b.n, x, nu[nu.index], tau.n[nu.index])
        }  
      }
    }

    max(area)
  } # end of residual.prob.max


  SmallSlice <- function(a, b, left, nu, tau.n)
  {
    # a, b      : numeric(2)
    # left      : logical(1)
    # nu, tau.n : numeric(1)

    x.star <- -diff(a)/diff(b)
    y.star <- a[1] + b[1]*x.star

    opposite.slices <- ratio.of.two.independent.normal.distrns.cdf(b, -y.star, nu-x.star, 1, tau.n)
    mu <- -a - b*nu ; sd <- sqrt(1 + b^2/tau.n)
    p <- pnorm(0, mean=mu, sd=sd, lower.tail=F)
    above.and.right <- p[1]
    above.and.left  <- p[2]

    side <- ifelse(left, -1, 1)
    area <- (opposite.slices + side*(above.and.right - above.and.left)) /2
    area
  } # end of SmallSlice 


  TinyBite <- function(a, b, k, nu, tau.n)
  {
    # a, b      : numeric(2)
    # k         : numeric(1)
    # nu, tau.n : numeric(1)

    x.intersect <- -diff(a)/diff(b)
    left <- k < x.intersect

    area <- SmallSlice(a, b, left, nu, tau.n) - LargeBite(a, b, k, nu, tau.n)
    area
  } # end of TinyBite


  TotArea <- function(an.matrix, b.n, sqrt.prec.n, nu, tau.n, prior1.mixture.wt, both)
  {
    intersections <- x.Intersections(an.matrix, b.n, sqrt.prec.n, nu, tau.n)

    b <- b.n[2]/b.n[1]
    a <- an.matrix[,2] - b*an.matrix[,1]
    xt.zero <- a/(1-b)
    x.zero <- -(an.matrix[,1] - xt.zero)/b.n[1]

    nu.len <- length(nu)
    area <- numeric(nu.len)
    nu.indices <- seq(nu.len)
    j.indices <- which(intersections$from < intersections$to)

    for (j.index in j.indices)
    {
      j <- intersections$j[j.index]
      tmp.interval <- c(intersections$from[j.index], intersections$to[j.index])

      if (is.infinite(tmp.interval[1]))
      {
        if (tmp.interval[2] < x.zero[j])
        {
          for (nu.index in nu.indices)
          {
            area[nu.index] <- area[nu.index] +  
                              LargeBite(an.matrix[j,], b.n, tmp.interval[2], nu[nu.index], tau.n[nu.index]) 
          }
        }
        else
        {
          for (nu.index in nu.indices)
          {
            area[nu.index] <- area[nu.index] + 
                              SmallSlice(an.matrix[j,], b.n, T, nu[nu.index], tau.n[nu.index]) +
                              TinyBite(an.matrix[j,], b.n, tmp.interval[2], nu[nu.index], tau.n[nu.index]) 
          }
        }
      }
      else if (is.infinite(tmp.interval[2]))
      {
        if (tmp.interval[1] < x.zero[j])
        {
          for (nu.index in nu.indices)
          {
            area[nu.index] <- area[nu.index] +
                              TinyBite(an.matrix[j,], b.n, tmp.interval[1], nu[nu.index], tau.n[nu.index]) +
                              SmallSlice(an.matrix[j,], b.n, F, nu[nu.index], tau.n[nu.index])
          }
        }
        else
        {
          for (nu.index in nu.indices)
          {
            area[nu.index] <- area[nu.index] +
                              LargeBite(an.matrix[j,], b.n, tmp.interval[1], nu[nu.index], tau.n[nu.index])
          }
        }
      }
      else if (prod(sign(tmp.interval-x.zero[j])) < 0)
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- area[nu.index] +
                            TinyBite(an.matrix[j,], b.n, tmp.interval[1], nu[nu.index], tau.n[nu.index]) +
                            TinyBite(an.matrix[j,], b.n, tmp.interval[2], nu[nu.index], tau.n[nu.index])
        }
      }
      else
      {
        for (nu.index in nu.indices)
        {
          area[nu.index] <- area[nu.index] +
                            DiamondBite(an.matrix[j,], b.n, tmp.interval, nu[nu.index], tau.n[nu.index])
        }
      }
    }


    if (both)
    {
      area <- max(area)
    }
    else
    {
      if (nu.len == 2) prior1.mixture.wt <- c(prior1.mixture.wt, 1-prior1.mixture.wt)
      area <- sum(prior1.mixture.wt*area)
    }

    area  
  } # end of TotArea


  x.Intersections <- function(a.n, b.n, sqrt.prec.n, nu, tau.n, stop.prob=1e-8)
  {
    j.Section <- function(j, x, cutoffs){1 + sum(x>=cutoffs[j,])}

    add.intersection <- function(my.list, j, last.x, new.x)
    {
      my.list$j <- c(my.list$j, j)
      my.list$from <- c(my.list$from, last.x)
      my.list$to <- c(my.list$to, new.x)
      my.list
    } # end of add.intersection
    
    # Compute stop-limits (useful for sections 1 & 4 to avoid infinite search intervals)
    
    stop.lim <- numeric(2)
    stop.lim[1] <- min(qnorm(log(stop.prob), mean=nu, sd=1/sqrt(tau.n), log.p=T))
    stop.lim[2] <- max(qnorm(log(stop.prob), mean=nu, sd=1/sqrt(tau.n), log.p=T, lower.tail=F))

    # Compute x.opt et x.zero for each z_j

    b <- b.n[2]/b.n[1]
    a <- a.n[,2] - b*a.n[,1]

    xt.zero <- a/(1-b)
    x.zero <- -(a.n[,1] - xt.zero)/b.n[1]

    xt.opt.1 <- (-2*a*b - sqrt((2*a*b)^2 -4*(b^2-1)*(a^2-2*log(b))))/(2*(b^2-1))
    xt.opt.2 <- (-2*a*b + sqrt((2*a*b)^2 -4*(b^2-1)*(a^2-2*log(b))))/(2*(b^2-1))
    xt.opt <- matrix(c(xt.opt.1, xt.opt.2), ncol=2)
    x.opt  <- -(a.n[,1] - xt.opt) /b.n[1]
    x.opt.min <- apply(x.opt, 1, min)
    x.opt.max <- apply(x.opt, 1, max)
    x.opt <- matrix(c(x.opt.min, x.opt.max), ncol=2)
    x.cutoffs <- matrix(c(x.opt.min, x.zero, x.opt.max), ncol=3)

    # Compute height at each opt point

    h <- matrix(NA, nrow=nrow(a.n), ncol=2)
    bn.matrix <- matrix(b.n, nrow=nrow(a.n), ncol=2, byrow=T)

    j <- 1
    u <- a.n + bn.matrix*x.opt[,j]
    u1 <- pmax(u[,1], u[,2])
    u2 <- pmin(u[,1], u[,2])
    h[,j] <- pnorm.diff.log(u1,u2)

    j <- 2
    u <- a.n + bn.matrix*x.opt[,j]
    u1 <- pmax(u[,1], u[,2])
    u2 <- pmin(u[,1], u[,2])
    h[,j] <- pnorm.diff.log(u1,u2)

    h.max <- apply(h, 1, max)
    j <- which.max(h.max)
    j.col <- which.max(h[j,])
    j.section <- ifelse(j.col==1, 2, 4)

    found.limit.j <- rep(F, 2)
    j0 <- j
    j0.section <- j.section
    x0 <- x.opt[j, j.col]
    current.x <- x0

    Intersections <- list(j=numeric(0), from=numeric(0), to=numeric(0))

    # Find intersections to the right of global max

    residual.prob <- residual.prob.max(a.n, b.n, current.x, F, nu, tau.n)
    continue <- residual.prob > stop.prob

    while (continue)
    {
      tmp <- next.intersection(j, j.section, a.n, b.n, x.cutoffs, current.x, left2right=T, stop.lim)

      if (length(tmp$j) == 0)
      {
        if (j.section == 4)
        {
          found.limit.j[2] <- T
          Intersections <- add.intersection(Intersections, j, current.x, Inf)
        }
        else
        {
          new.x <- x.cutoffs[j,j.section]
          Intersections <- add.intersection(Intersections, j, current.x, new.x)
          current.x <- new.x
          j.section <- j.section + 1
          residual.prob <- residual.prob.max(a.n, b.n, new.x, F, nu, tau.n)
        }
      }
      else
      { 
        new.j <- tmp$j
        new.x <- tmp$x
        Intersections <- add.intersection(Intersections, j, current.x, new.x)
        j <- new.j
        j.section <- j.Section(j, new.x, x.cutoffs)
        current.x <- new.x
        residual.prob <- residual.prob.max(a.n, b.n, new.x, F, nu, tau.n)
      }

      continue <- !found.limit.j[2] & residual.prob > stop.prob
    }

    # Find intersections to the left of global max

    if (j.col == 1)
    {
      found.limit.j[1] <- T
      Intersections <- add.intersection(Intersections, j0, -Inf, x0)
    }
    else
    { 
      j <- j0
      j.section <- j0.section - 1
      current.x <- x0
      residual.prob <- residual.prob.max(a.n, b.n, current.x, T, nu, tau.n)
      continue <- residual.prob > stop.prob

      while (continue)
      {
        tmp <- next.intersection(j, j.section, a.n, b.n, x.cutoffs, current.x, left2right=F, stop.lim)

        if (length(tmp$j) == 0)
        {
          if (j.section == 1)
          {
            found.limit.j[1] <- T
            Intersections <- add.intersection(Intersections, j, -Inf, current.x)
          }
          else
          {
            new.x <- x.cutoffs[j, j.section-1]
            Intersections <- add.intersection(Intersections, j, new.x, current.x)
            current.x <- new.x
            j.section <- j.section - 1
            residual.prob <- residual.prob.max(a.n, b.n, new.x, T, nu, tau.n)
          }
        }
        else
        { 
          new.j <- tmp$j
          new.x <- tmp$x
          Intersections <- add.intersection(Intersections, j, new.x, current.x)
          j <- new.j
          j.section <- j.Section(j, new.x, x.cutoffs)
          current.x <- new.x
          residual.prob <- residual.prob.max(a.n, b.n, new.x, T, nu, tau.n) 
        }

        continue <- !found.limit.j[1] & residual.prob > stop.prob
      }
    }

    # Sort intersection elements in ascending order

    o <- order(Intersections$from)
    Intersections$j <- Intersections$j[o]
    Intersections$from <- Intersections$from[o]
    Intersections$to <- Intersections$to[o]

    # Merge consecutive sections from same z_j

    drop.elem <- function(x, k){x[-k]}
    elem <- function(x, k){x[k]}

    tmp.Intersections <- Intersections
    Intersections <- lapply(Intersections, elem, 1)
    tmp.Intersections <- lapply(tmp.Intersections, drop.elem, 1)

    j.indices <- seq(along=tmp.Intersections$j)
    for (j.index in j.indices)
    {
      j    <- tmp.Intersections$j[j.index]
      from <- tmp.Intersections$from[j.index]
      to   <- tmp.Intersections$to[j.index]

      l <- length(Intersections$j)
      if (j == Intersections$j[l])
      {
        Intersections$to[l] <- to
      }
      else
      {
        Intersections$j <- c(Intersections$j, j)
        Intersections$from <- c(Intersections$from, from) 
        Intersections$to <- c(Intersections$to, to)
      }
    }

    Intersections$from[1] <- -Inf
    Intersections$to[length(Intersections$to)] <- Inf

    Intersections
  } # end of x.Intersections
  

  # .... Start function ........................................................

  # Initial values

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  

  out <- list(continue=T, target=accepted.cdf.diff, increasing.outcome.with.n=F, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    z <- g.mean(n, cdf.points, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)

    outcome <- c(z$res, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  
  
  
  if (z$need.convexity.check)
  {
    tmp <- ab.parms(n, cdf.points, prec, mu0s, prec0s, tau)
    intersections <- x.Intersections(tmp$a.n, tmp$b.n, tmp$sqrt.prec.n, nu, tmp$tau.n)
    sd <- 1/sqrt(tmp$tau.n)
    
    # Generate x.bar from clinical prior
    if (length(nu) == 2)
    {
      if (both) prior1.mixture.wt <- 0.5
      w <- sample(c(1,2), convexity.check.sim.size, replace=T, prob=c(prior1.mixture.wt, 1-prior1.mixture.wt))
      x.bar <- rnorm(convexity.check.sim.size, mean=nu[w], sd=sd[w]) 
    }
    else
    {
      x.bar <- rnorm(convexity.check.sim.size, mean=nu, sd=sd)
    }
    
    # Compute j.max index for each sampled x.bar
    m <- length(cdf.points)
    w <- rep(seq(convexity.check.sim.size), rep(m, convexity.check.sim.size))
    u1 <- tmp$a.n[,1] + tmp$b.n[1]*x.bar[w]
    u2 <- tmp$a.n[,2] + tmp$b.n[2]*x.bar[w]
    pdiff <- pnorm.diff.log(u1, u2)
    pdiff <- matrix(pdiff, byrow=T, ncol=m)
    j.max <- apply(pdiff, 1, which.max)
    
    # Compare it to j indices obtained from calculations based on quasi-log-convexity asumption
    from <- matrix(intersections$from, byrow=T, ncol=length(intersections$from), nrow=convexity.check.sim.size)
    xbar.which.interval <- apply(x.bar > from, 1, sum)
    xbar.jmax.from.logConvexity.asumption <- intersections$j[xbar.which.interval]
    convexity.check.success.rate <- mean(j.max == xbar.jmax.from.logConvexity.asumption)
  }
  else
  {
    convexity.check.sim.size <- NA
    convexity.check.success.rate <- NA
  }

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome), convexity.check.sim.size=convexity.check.sim.size, convexity.check.success.rate=convexity.check.success.rate)
} # end of .ss.cons.norm.knownvar.avg.cdf


.ss.cons.norm.knownvar.avg.hpdlimits <- function(accepted.diff, prec,
                                                           prior1, prior2, 
                                                           level, n.max=100000, next.n=.ss.nextn4nonrndoutcomes,
                                                           clinical.prior=list(),
                                                           prior1.mixture.wt=1, both=F) 
{                     
  mu0s   <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  
  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
  }
  else
  {
    nu  <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }

               
  h.mean <- function(n, level, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
  {
    z <- qnorm((1+level)/2)

    if (n == 0)
    {
      res <- max(abs(diff(mu0s)+z*diff(1/sqrt(prec0s))), abs(diff(mu0s)-z*diff(1/sqrt(prec0s))))
    }
    else
    {
      prec.n <- prec0s + n*prec
      abs.Cn <- abs(z*diff(1/sqrt(prec.n)))
      B.n <- n*prec*diff(1/prec.n)
      A.n <- diff(prec0s*mu0s/prec.n)
      tau.n <- n*tau*prec/(tau+n*prec) # x.bar ~ N(nu, tau.n)
      res <- abs.Cn + abs(B.n) * .ss.cons.absNorm.mean(nu+A.n/B.n, tau.n, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    
    return(res)
  } # end of h.mean


  # Initial values

  n.start <- .ss.cons.norm.knownvar.nStart(prec, accepted.diff, mu0s, prec0s, prior1.mixture.wt=prior1.mixture.wt, both=both, nu=clinical.prior$mu0, tau=clinical.prior$prec0)

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=accepted.diff, increasing.outcome.with.n=F, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    z <- h.mean(n, level, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.knownvar.avg.hpdlimits


.ss.cons.norm.knownvar.prob.hpdlimits <- function(accepted.diff, prec,
                                                            prior1, prior2, 
                                                            prob, level, n.start=1000, n.max=100000,
                                                            next.n=.ss.nextn4nonrndoutcomes, 
                                                            clinical.prior=list(),
                                                            prior1.mixture.wt=1, both=F)
{                     
  mu0s   <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  
  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
  }
  else
  {
    nu  <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }


  h.prob <- function(n, accepted.diff, level, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
  {
    z <- qnorm((1+level)/2)               
    prec.n <- prec0s + n*prec
    abs.Cn <- abs(z*diff(1/sqrt(prec.n)))
    B.n <- n*prec*diff(1/prec.n)
    A.n <- diff(prec0s*mu0s/prec.n)
    tau.n <- n*tau*prec/(tau+n*prec) # x.bar ~ N(nu, tau.n)
    res <- .ss.cons.absNorm.cdf(abs(B.n), nu+A.n/B.n, tau.n, abs.Cn, accepted.diff, prior1.mixture.wt=prior1.mixture.wt, both=both)

    res
  } # end of h.prob
 
 
  # Initial values
  
  n.start <- .ss.cons.norm.knownvar.nStart(prec, accepted.diff, mu0s, prec0s, prior1.mixture.wt=prior1.mixture.wt, both=both, nu=clinical.prior$mu0, tau=clinical.prior$prec0, p=prob)                                                             
  
  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=prob, increasing.outcome.with.n=T, step0=step0, n.max=n.max)

  while (out$continue)
  {
    z <- h.prob(n, accepted.diff, level, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
    
    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.knownvar.prob.hpdlimits


.ss.cons.norm.knownvar.prob.cdf <- function(cdf.points, accepted.cdf.diff, prec,
                                                      prior1, prior2,  
                                                      n.start, n.max, prob, next.n=.ss.nextn4nonrndoutcomes, 
                                                      clinical.prior=list(),
                                                      prior1.mixture.wt=1, both=F, epsilon=1e-8)
{
  xbar.intervals <- function(accepted.cdf.diff, a, b, epsilon)
  {
    find.xopt <- function(accepted.cdf.diff, a, b, x0, x1, epsilon)
    {
      h0 <- abs(pnorm(a[1]-b[1]*x0) - pnorm(a[2]-b[2]*x0))
      h1 <- abs(pnorm(a[1]-b[1]*x1) - pnorm(a[2]-b[2]*x1))
      positive.slope <- h1 > h0
    
      continue <- T
      next.x <- (x0 + x1)/2
    
      while (continue)
      {
        x <- next.x
        h <- abs(pnorm(a[1]-b[1]*x) - pnorm(a[2]-b[2]*x))
        h.higher.than.target <- h > accepted.cdf.diff
      
        if (xor(h.higher.than.target, positive.slope))
        {
          x0 <- x
        }
        else
        {
          x1 <- x
        }
      
        next.x <- (x0 + x1)/2
        continue <- abs(x-next.x) > epsilon
      }
    
      x
    } # end of find.xopt

    
    B <- b[2]/b[1]
    A <- a[2] - B*a[1]
    
    if (B == 1)
    {
      z.opt <- -A/2
      h <- abs(pnorm(z.opt)-pnorm(A+B*z.opt))
      
      if (h <= accepted.cdf.diff)
      {
        interval <- c(-Inf, Inf)
      }
      else
      {
        leftside.zsoln.when.B.eq.1 <- function(A, accepted.cdf.diff, epsilon)
        {
          z.opt <- -A/2
          target.pdiff <- -sign(A)*accepted.cdf.diff
          p.lim <- numeric(2)
          z.lim <- numeric(2)
          
          # Set right-side-limits
          p.lim[2] <- pnorm(z.opt)
          z.lim[2] <- z.opt
          
          # Set left-side limits
          if (A < 0)
          {
            p.lim[1] <- accepted.cdf.diff
            z.lim[1] <- qnorm(p.lim[1])
          }
          else
          {
            z.lim[1] <- qnorm(accepted.cdf.diff) - A
            p.lim[1] <- pnorm(z.lim[1])  
          }
          
          # Hybrid bisectional/Newton-Raphson algorithm
          p <- mean(p.lim)
          z <- qnorm(p)
          continue <- T
          
          while (continue)
          {
            z.old <- z
            g <- pnorm(z) - pnorm(A+z) - target.pdiff
            
            if (xor(g>0, A>0))
            {
              z.lim[2] <- z
              p.lim[2] <- pnorm(z)
            }
            else
            {
              z.lim[1] <- z
              p.lim[1] <- pnorm(z)
            }
            
            g.prime <- dnorm(z) - dnorm(A+z)
            z <- z - g/g.prime
            
            # If Newton-Raphson (above) throws us out of bounds, then perform a bisectional step instead
            if (z<=z.lim[1] | z>= z.lim[2])
            {
              p <- mean(p.lim)
              z <- qnorm(p)
            }
            
             continue <- abs(z-z.old) > epsilon
          } # end of while-continue loop
          
          z
        } # end of leftside.zsoln.when.B.eq.1
        
        z.left <- leftside.zsoln.when.B.eq.1(A, accepted.cdf.diff, epsilon)
        z.right <- 2*z.opt - z.left # by symmetry
        z <- c(z.left, z.right)
        x <- (a[1]-z)/b[1]
        interval <- c(-Inf, sort(x), Inf)
        interval <- matrix(interval, byrow=T, ncol=2)
      }
    }
    else
    {
      delta <- (2*A*B)^2 - 4*(B*B-1)*(A*A-2*log(B))

      num1 <- -2*A*B
      denom <- 2*(B*B-1)

      z.opt <- sort((num1 + c(-1, 1) * sqrt(delta))/denom)
      z.low <- A/(1-B)
      z.opt <- sort(c(z.opt, z.low))
    
    
      h <- abs(pnorm(z.opt)-pnorm(A+B*z.opt))
  
      if (max(h) <= accepted.cdf.diff)
      {
        interval <- c(-Inf, Inf)
      }
      else
      {
        x.opt <- sort((a[1]-z.opt)/b[1])
        h <- pnorm(a[1]-b[1]*x.opt) - pnorm(a[2]-b[2]*x.opt)
        h.sign <- sign(h)
        which.above <- which(abs(h) > accepted.cdf.diff)

        if (length(which.above) == 3)
        {
          # leftmost local max
          z.min <- qnorm(1-accepted.cdf.diff)

          if (h.sign[1] > 0)
          {
            x.min <- (a[1]-z.min)/b[1]
          }
          else
          {
            x.min <- (a[2]-z.min)/b[2]
          }
          x.opt.left <- find.xopt(accepted.cdf.diff, a, b, x.min, x.opt[1], epsilon)
      
          # rightmost local max
          z.max <- qnorm(accepted.cdf.diff)

          if (h.sign[3] > 0)
          {
            x.max <- (a[1]-z.max)/b[1]
          }
          else
          {
            x.max <- (a[2]-z.max)/b[2]
          }
          x.opt.right <- find.xopt(accepted.cdf.diff, a, b, x.opt[3], x.max, epsilon)

          interval <- c(-Inf, x.opt.left, x.opt.right, Inf)
        }
        else if (length(which.above) == 2)
        {
          # around leftmost local max
          z.min <- qnorm(1-accepted.cdf.diff)

          if (h.sign[1] > 0)
          {
            x.min <- (a[1]-z.min)/b[1]
          }
          else
          {
            x.min <- (a[2]-z.min)/b[2]
          }
          x.opt.right1 <- find.xopt(accepted.cdf.diff, a, b, x.opt[1], x.opt[2], epsilon)
          x.opt.left1  <- find.xopt(accepted.cdf.diff, a, b, x.min, x.opt[1], epsilon)
      
          # around rightmost local max
          z.max <- qnorm(accepted.cdf.diff)

          if (h.sign[3] > 0)
          {
            x.max <- (a[1]-z.max)/b[1]
          }
          else
          {
            x.max <- (a[2]-z.max)/b[2]
          }
          x.opt.right2 <- find.xopt(accepted.cdf.diff, a, b, x.opt[3], x.max, epsilon)
          x.opt.left2  <- find.xopt(accepted.cdf.diff, a, b, x.opt[2], x.opt[3], epsilon)
      
          interval <- c(-Inf, x.opt.left1, x.opt.right1, x.opt.left2, x.opt.right2, Inf)
        }
        else
        {
          # There is only one point above target
      
          if (which.above == 1)
          {
            z.min <- qnorm(1-accepted.cdf.diff)
        
            if (h.sign[1] > 0)
            {
              x.min <- (a[1]-z.min)/b[1]
            }
            else
            {
              x.min <- (a[2]-z.min)/b[2]
            }
            x.opt.right <- find.xopt(accepted.cdf.diff, a, b, x.opt[1], x.opt[2], epsilon)
            x.opt.left  <- find.xopt(accepted.cdf.diff, a, b, x.min, x.opt[1], epsilon)
            interval <- c(-Inf, x.opt.left, x.opt.right, Inf)
          }
          else
          {
            # which.above == 3
            z.max <- qnorm(accepted.cdf.diff)

            if (h.sign[3] > 0)
            {
              x.max <- (a[1]-z.max)/b[1]
            }
            else
            {
              x.max <- (a[2]-z.max)/b[2]
            }
            x.opt.right <- find.xopt(accepted.cdf.diff, a, b, x.opt[3], x.max, epsilon)
            x.opt.left  <- find.xopt(accepted.cdf.diff, a, b, x.opt[2], x.opt[3], epsilon)
            interval <- c(-Inf, x.opt.left, x.opt.right, Inf)
          }
        }
      } 
  
      interval <- matrix(interval, byrow=T, ncol=2)
    } # end of B<>1 conditional code
    
    interval
  } # end of xbar.intervals
  
  
  combined.intervals <- function(intervals)
  {
    cdf.points <- unique(intervals[,1])
    
    q1 <- cdf.points[1]
    w <- which(intervals[,1]==q1)
    out.intervals <- intervals[w,,drop=F][,-1,drop=F]
    
    for (q in cdf.points[-1])
    {
      w <- which(intervals[,1]==q)
      tmp.intervals <- intervals[w,,drop=F][,-1,drop=F]
      m <- nrow(tmp.intervals)
      w <- rep(seq(m), rep(nrow(out.intervals), m))
      intervals.lower.limits <- pmax(out.intervals[,1], tmp.intervals[,1][w])
      intervals.upper.limits <- pmin(out.intervals[,2], tmp.intervals[,2][w])
      w <- which(intervals.lower.limits < intervals.upper.limits)
      out.intervals <- matrix(c(intervals.lower.limits[w], intervals.upper.limits[w]), ncol=2)
    }
    
    out.intervals
  } # end of combined.intervals


  mu0s     <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)

  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
    prior1.mixture.wt <- c(prior1.mixture.wt, 1-prior1.mixture.wt)
  }
  else
  {
    nu <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }
  

  # Initial values
  
  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  
  out <- list(target=prob, increasing.outcome.with.n=T, step0=step0, n.max=n.max)
  continue <- T
 
  while (continue)
  {
    if (n == 0)
    {
      p01 <- pnorm(cdf.points, prior1$mu0, sd=1/sqrt(prior1$prec0))
      p02 <- pnorm(cdf.points, prior2$mu0, sd=1/sqrt(prior2$prec0))
      pdiff <- abs(p01-p02)
      p <- as.double(all(pdiff<=accepted.cdf.diff))
    }
    else
    {
      prec.n <- prec0s + n*prec
      a.intercept <- -prec0s*mu0s/sqrt(prec.n)
      a.slope <- sqrt(prec.n)
      b <- n*prec/sqrt(prec.n)
    
      intervals <- numeric(0)
      for (Q in cdf.points)
      {
        a <- a.intercept + a.slope * Q
        tmp.intervals <- xbar.intervals(accepted.cdf.diff, a, b, epsilon)
        tmp.intervals <- cbind(Q, tmp.intervals)
        intervals <- rbind(intervals, tmp.intervals)
      }
      
      # Combine intervals
      intervals <- combined.intervals(intervals) 
       
      # Compute total probability of the set of disjoint intervals obtained above
    
      if (length(nu) == 1)
      {
        w <- 1
      }
      else
      {
        w <- c(1, 2)
      }
    
      if (length(intervals) == 0)
      {
        p <- 0
      }
      else
      {
        tau.n <- n*tau*prec/(tau+n*prec)
        
        if (length(nu) == 2)
        {
          # replicate each line of intervals twice (once for each specialist prior distrn)
          m <- nrow(intervals)
          w <- rep(seq(m), rep(2, m))
          intervals <- intervals[w,]
        }
        
        p <- pnorm(intervals, mean=nu, sd=1/sqrt(tau.n))
        p <- p[,2] - p[,1]
        p <- matrix(p, byrow=T, ncol=length(nu))
        p <- apply(p, 2, sum) 
                       
        p <- ifelse(both, min(p), sum(prior1.mixture.wt*p))
      }
    }
      
    outcome <- c(p, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited
    
    nextn <- next.n(out)
    continue <- nextn$continue
    n <- nextn$n
  }
  
  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.knownvar.prob.cdf


.ss.cons.norm.knownvar.avg.q <- function(quantiles, accepted.diff, prec,
                                                           prior1, prior2, prior1.mixture.wt=1, clinical.prior=list(), both=F,
                                                           n.max=100000, next.n=.ss.nextn4nonrndoutcomes)
{                     
  mu0s   <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  
  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
  }
  else
  {
    nu  <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }


  qdiff.mean <- function(n, quantiles, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
  {
    if (length(quantiles) > 1) quantiles <- c(min(quantiles), max(quantiles))
    z <- qnorm(quantiles)

    if (n == 0)
    {
      res <- abs(diff(mu0s)+z*diff(1/sqrt(prec0s)))
      res <- max(res)
    }
    else
    {
      prec.n <- prec0s + n*prec
      B.n <- n*prec*diff(1/prec.n)
      D.n <- diff(prec0s*mu0s/prec.n) + z*diff(1/sqrt(prec.n))
      tau.n <- n*tau*prec/(tau+n*prec) # x.bar ~ N(nu, tau.n)
      V <- .ss.cons.norm.Vshape.coords(B.n, D.n)
      res <- V$h0 + V$slope * .ss.cons.absNorm.mean(nu - V$x.min, tau.n, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    
    return(res)
  } # end of qdiff.mean


  # Initial values

  n.start <- .ss.cons.norm.knownvar.nStart(prec, accepted.diff, mu0s, prec0s, nu=clinical.prior$mu0, tau=clinical.prior$prec0, prior1.mixture.wt=prior1.mixture.wt, both=both)   

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=accepted.diff, increasing.outcome.with.n=F, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    z <- qdiff.mean(n, quantiles, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.knownvar.avg.q


.ss.cons.norm.knownvar.prob.q <- function(quantiles, accepted.diff, prec,
                                                            prior1, prior2, clinical.prior=list(), prior1.mixture.wt=1, both=F,
                                                            prob, n.max=100000, next.n=.ss.nextn4nonrndoutcomes)
{                     
  mu0s   <- c(prior1$mu0, prior2$mu0)
  prec0s <- c(prior1$prec0, prior2$prec0)
  
  if (!.ss.cons.defined.list(clinical.prior))
  {
    nu  <- mu0s
    tau <- prec0s
  }
  else
  {
    nu  <- clinical.prior$mu0
    tau <- clinical.prior$prec0
    prior1.mixture.wt <- 1
  }


  qdiff.prob <- function(n, accepted.diff, quantiles, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)
  {
    if (length(quantiles) > 1) quantiles <- c(min(quantiles), max(quantiles))
    z <- qnorm(quantiles)

    if (n == 0)
    {
      res <- abs(diff(mu0s)+z*diff(1/sqrt(prec0s)))
    }
    else
    {
      prec.n <- prec0s + n*prec
      B.n <- n*prec*diff(1/prec.n)
      D.n <- diff(prec0s*mu0s/prec.n) + z*diff(1/sqrt(prec.n))
      tau.n <- n*tau*prec/(tau+n*prec) # x.bar ~ N(nu, tau.n)
      V <- .ss.cons.norm.Vshape.coords(B.n, D.n)
      res <- .ss.cons.absNorm.cdf(V$slope, nu-V$x.min, tau.n, V$h0, accepted.diff, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }

    res
  } # end of qdiff.prob


  # Initial values

  n.start <- .ss.cons.norm.knownvar.nStart(prec, accepted.diff, mu0s, prec0s, prior1.mixture.wt=prior1.mixture.wt, both=both, nu=clinical.prior$mu0, tau=clinical.prior$prec0, p=prob)  

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=prob, increasing.outcome.with.n=T, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    z <- qdiff.prob(n, accepted.diff, quantiles, prec, mu0s, prec0s, prior1.mixture.wt, both, nu, tau)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }  

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.norm.knownvar.prob.q


.ss.nextn4nonrndoutcomes <- function(mylist)
{
  ##################################################################
  # mylist: consists in a list of the following arguments:
  # -------
  #
  # n.visited
  # outcome
  # target
  # step0
  # increasing.outcome.with.n
  # n.max  
  #
  ##################################################################

  continue <- T

  sufficient.n <- (mylist$increasing.outcome.with.n & mylist$outcome >= mylist$target) | (!mylist$increasing.outcome.with.n & mylist$outcome <= mylist$target)
  n.visited <- mylist$n.visited
  
  if (all(sufficient.n))
  {
    n.visited <- sort(n.visited)
    
    if (n.visited[1] == 0)
    {
      n <- 0
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- max(n.visited - mylist$step0, 0)
      }
      else
      {
        n <- max(n.visited[1] - 2*diff(n.visited[1:2]), 0)
      }
    }
  }
  else if (all(!sufficient.n))
  {
    n.visited <- rev(sort(n.visited))
    
    if (n.visited[1] == mylist$n.max)
    {
      n <- Inf
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- n.visited + mylist$step0
      }
      else
      {
        n <- n.visited[1] + 2 * (n.visited[1]-n.visited[2])
      }
    
      n <- min(n, mylist$n.max)
    }
  }
  else
  {
    smallest.sufficient.n  <- min(n.visited[sufficient.n])
    largest.insufficient.n <- max(n.visited[!sufficient.n])
    
    if ((smallest.sufficient.n-largest.insufficient.n) == 1)
    {
      n <- smallest.sufficient.n
      continue <- F
    }
    else
    {
      n <- floor((smallest.sufficient.n + largest.insufficient.n)/2)
    }
  }

  # If continue=F, then n indicates the optimal sample size found

  list(n=n, continue=continue)
} # end of .ss.nextn4nonrndoutcomes


.ss.bsearch <- function(mylist)
{
  ##################################################################
  # mylist: consists in a list of the following arguments:
  # -------
  #
  # n.visited
  # outcome
  # nmax
  # mcs
  # min.for.possible.return
  # target
  # step0
  # increasing.outcome.with.n  
  #
  ##################################################################

  all.same.dir <- function(nvisited){length(unique(sign(diff(nvisited)))) == 1}

  #

  last.outcome <- mylist$outcome[1]
  dir <- ifelse(last.outcome==mylist$target, -1,
                ifelse(mylist$increasing.outcome.with.n, -1, 1)*sign(last.outcome-mylist$target))
  continue <- T

  if (length(mylist$n.visited) == 1)
  {
    n <- ifelse(dir == 1, mylist$n.visited + dir * mylist$step0, floor(mylist$n.visited/2))
    n <- min(max(0, n), mylist$nmax)
  }
  else
  {
    steps <- -diff(mylist$n.visited)

    if (abs(steps[1]) == 1 && length(unique(sign(mylist$outcome[1:2]-mylist$target))) == 2)
    {
      cond <- ifelse(rep(mylist$increasing.outcome.with.n, 2),
                     mylist$outcome[1:2] >= mylist$target, mylist$outcome[1:2] <= mylist$target)
      n <- min(mylist$n.visited[1:2][cond])
      continue <- F
    }
    else if (length(mylist$n.visited) > mylist$mcs && all(mylist$n.visited[1:(mylist$mcs+1)]==mylist$nmax) && dir == 1)
    {
      n <- mylist$nmax
      continue <- F
    }
    else if (length(mylist$n.visited) > mylist$mcs && all(mylist$n.visited[1:(mylist$mcs+1)]==0) && dir == -1)
    {
      n <- 0
      continue <- F
    }
    else if (mylist$n.visited[1] == max(mylist$n.visited) && dir == 1 && abs(steps[1]) != 1)
    {
      n <- mylist$n.visited[1] + 2*max(abs(diff(mylist$n.visited)))
      n <- min(n, mylist$nmax)
    }
    else if (mylist$n.visited[1] == min(mylist$n.visited) && dir == -1 && abs(steps[1]) != 1)
    {
      n <- floor(mylist$n.visited[1]/2)
    }
    else
    {
      possible.to.move.back <- abs(steps[1]) > mylist$min.for.possible.return

      if (possible.to.move.back && length(mylist$n.visited) > mylist$mcs && all.same.dir(mylist$n.visited[1:(mylist$mcs+1)]) && sign(steps[1]) == dir && 
          (length(mylist$n.visited) == (mylist$mcs+1) || !all.same.dir(mylist$n.visited[1:(mylist$mcs+2)])) )
      {
        n <- mylist$n.visited[mylist$mcs+2]
      }
      else
      {
        if (abs(steps[1]) == 1)
        {
          m <- mylist$n.visited[1] + dir
        }
        else
        {
          d <- mylist$n.visited - mylist$n.visited[1]
          m <- mylist$n.visited[sign(d)==dir][1]
          m <- ceiling((mylist$n.visited[1]+m)/2)
        }

        if (dir == 1)
        {
          n <- max(m, mylist$n.visited[1]+1)
        }
        else
        {
          n <- min(m, mylist$n.visited[1]-1)
        }

        n <- min(n, mylist$nmax)
        n <- max(0, n)
      }
    }
  }

  # If continue=F, then n indicates the optimal sample size found

  list(n=n, continue=continue)
} # end of .ss.bsearch


.ss.cons.norm.check.list <- function(test.prior, prior.name, known.var=F)
{
  if (missing(test.prior))
  {
    stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F) 
  }
  else
  {
    tmp <- try(paste(test.prior), silent=T)
    if (class(tmp) != "try-error")
    {
      dim.names <- names(test.prior)
      
      if (known.var)
      {
        present.dim <- match(c("mu0", "prec0"), dim.names)
        if (any(is.na(present.dim))) stop(paste(c("Both mu0 and prec0 must be dimensions of ", prior.name, " list."), collapse=''), call.=F)
      }
      else
      {
        present.dim <- match(c("mu0", "n0", "prec.shape", "prec.rate"), dim.names)
        if (any(is.na(present.dim))) stop(paste(c("Dimensions mu0, n0, prec.shape and prec.rate must be part of ", prior.name, " list."), collapse=''), call.=F)
      }
    }
    else
    {
      stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F)
    }
  }
} # end of .ss.cons.norm.check.list


.ss.cons.norm.Vshape.coords <- function(common.slope, intercepts)
{
  x.intersects <- -intercepts/common.slope
  x.min <- mean(x.intersects)
  slope <- abs(common.slope)
  h0 <- ifelse(length(intercepts) == 2, slope*abs(diff(x.intersects))/2, 0)
  
  # V-height @ x.bar:  h0 + slope*|x.bar - x.min|
  
  list(slope=slope, x.min=x.min, h0=h0)
} # end of .ss.cons.norm.Vshape.coords


.ss.cons.absNorm.cdf <- function(b, mu, prec, r, d, prior1.mixture.wt=numeric(0), both=logical(0))
{
  # Return P{b*|N(mu,prec)| + |r| <= d}
  # mu and prec can be vectors of length 2
  # b is assumed positive

  d <- d - abs(r)
  mu.len <- length(mu)
  
  if (d < 0)
  {
    prob <- rep(0, mu.len)
  }
  else
  {
    ncp <- prec*(mu^2)
    q <- prec*((d/b)^2)
    prob <- pchisq(q, 1, ncp=ncp)
  }
  
  prob <- ifelse(!both & mu.len==2, sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*prob), min(prob))
  prob
} # end of .ss.cons.absNorm.cdf


.ss.cons.absNorm.mean <- function(mu, prec, prior1.mixture.wt=numeric(0), both=logical(0))
{
  # Returns E|N(mu,prec)|
  # mu and prec can be vectors of length 2

  pi <- 4 * atan(1)
  mean <- sqrt(2/(pi*prec))*exp(-prec*(mu^2)/2) + mu*(2*pnorm(sqrt(prec)*mu)-1)
  mean <- ifelse(!both & length(mu)==2, sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*mean), max(mean))
  mean
} # end of .ss.cons.absNorm.mean


.ss.cons.norm.PostMoments.sample <- function(n, sim.size, w, alpha.gen, beta.gen, mu0.gen, n0.gen, alphas, betas, mu0s, n0s)
{
  # Generate ns2 from marginal distribution:
  # ns2 is gamma-gamma, which needs two steps to be generated
  if (n > 1)
  {
    ns2 <- rgamma(sim.size, alpha.gen[w], 2*beta.gen[w])
    ns2 <- rgamma(sim.size, (n-1)/2, ns2)
  }
  else ns2 <- rep(0, sim.size)

  # Generate xbar from conditional marginal (on ns2)
  condmarg.alpha  <- n + 2*alpha.gen[w] - 1
  condmarg.prec <- condmarg.alpha * n * n0.gen[w] / (n+n0.gen[w]) / (ns2+2*beta.gen[w])
  xbar <- mu0.gen[w] + rt(sim.size, condmarg.alpha)/sqrt(condmarg.prec)

  # Posterior moments for mu, from two clinicians' points of view
  post.moments <- function(n, mu0, n0, alpha, beta, xbar, ns2)
  {
    mu.n     <- (n0*mu0 + n*xbar)/(n+n0)
    beta.n   <- beta + ns2/2 + n0*n*((mu0-xbar)^2)/2/(n0+n)
    lambda.n <- (n+n0)*(alpha+n/2)/beta.n
    alpha.n  <- 2*alpha + n
    list(mu.n=mu.n, lambda.n=lambda.n, alpha.n=alpha.n)
  } # end of post.moments

  pm1 <- post.moments(n, mu0s[1], n0s[1], alphas[1], betas[1], xbar, ns2)
  pm2 <- post.moments(n, mu0s[2], n0s[2], alphas[2], betas[2], xbar, ns2)
  
  list(pm1=pm1, pm2=pm2)
} # end of .ss.cons.norm.PostMoments.sample


.ss.cons.norm.knownvar.nStart <- function(lambda, accepted.diff, mu0s, lambda0s, p=numeric(0), nu=numeric(0), tau=numeric(0), prior1.mixture.wt=numeric(0), both=F)
{
  .ss.cons.norm.knownvar.nStart0 <- function(lambda, accepted.diff, mu0s, lambda0s, nu, tau, p=numeric(0))
  {
    nu.star <- nu - diff(lambda0s*mu0s)/diff(lambda0s)
  
    if (length(p) > 0)
    {
      ow <- options("warn")
      options(warn=-1)
      ncp <- tau*(nu.star^2)
      gamma2 <- tau*accepted.diff*accepted.diff/qchisq(p,df=1,ncp=ncp)
      gamma <- sqrt(gamma2)
      options(ow) # reset warning option
    }
    else
    {
      gamma <- accepted.diff/.ss.cons.absNorm.mean(nu.star, tau, both=F)
    }
  
    A <- lambda^2
    B <- lambda*sum(lambda0s)-lambda*abs(diff(lambda0s))/gamma
    C <- prod(lambda0s)
    n.solns <- (sqrt(B*B-4*A*C)-B)/(2*A)
    ceiling(n.solns)
  } # end of .ss.cons.norm.knownvar.nStart0
  
  
  if (both)
  {
    n1 <- .ss.cons.norm.knownvar.nStart0(lambda, accepted.diff, mu0s, lambda0s, mu0s[1], lambda0s[1], p=p)
    n2 <- .ss.cons.norm.knownvar.nStart0(lambda, accepted.diff, mu0s, lambda0s, mu0s[2], lambda0s[2], p=p)
    n0 <- max(n1, n2)
  }
  else if (length(nu) == 0)
  {
    n1 <- .ss.cons.norm.knownvar.nStart0(lambda, accepted.diff, mu0s, lambda0s, mu0s[1], lambda0s[1], p=p)
    n2 <- .ss.cons.norm.knownvar.nStart0(lambda, accepted.diff, mu0s, lambda0s, mu0s[2], lambda0s[2], p=p)
    n0 <- ceiling(exp(sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*log(c(n1,n2)))))
  }
  else
  {
    n0 <- .ss.cons.norm.knownvar.nStart0(lambda, accepted.diff, mu0s, lambda0s, nu, tau, p=p)
  }
  
  n0
} # end of .ss.cons.norm.knownvar.nStart


.ss.cons.defined.list <- function(mylist){ifelse(length(mylist) == 0, F, any(unlist(lapply(mylist, length)) > 0))}
