c***********************CONTENTS of frqsmp.f***************************
c
c FRQSMP       Sample size based on frequentist approach
c INVNOR       Inverse cdf Normal 0<p<=0.5
c CDFNOR       Cumulative distribution standard Normal (x>=0)
c PDFNOR       Probability density standard Normal
c POLYEV       Evaluation of polynomial
c
c**********************************************************************
c
c  Sample size calculation given CS and w for
c   binomial proportion.
c   Method based on frequentist approach:
c    n ~= \frac{z_{1-\alpha/2}^2(p*q)}{d^2} , where
c    z(.) is the abscissa value given the confidence CS,
c    p = binomial probability of success given by user, q=1-p
c    d = w/2
c    CS is credible set and w is length of interval
c
c**********************************************************************
c
      FUNCTION FRQSMP(p,CS,w)

c     ..Parameters..
      integer          izero
      double precision one,half
      parameter       (izero=0,one=1.0d0,half=0.5d0)

c     ..Arguments..
      integer          FRQSMP
      double precision p,q,CS,w

c     ..Local..
      double precision d,alpha,z

c     ..Calling procedures..
      double precision INVNOR         !inverse of cdf standard normal

      q = one - p                             !Probability of failure
      d = w*half
      d = d*d

      alpha = half*(one-CS)
      z     = INVNOR(alpha)                       !inverse cdf normal
      z     = z*z

      FRQSMP = nint((z*p*q/d)+half)

      RETURN
      END 
c
c**********************************************************************
c
c Abramowitz et al. equation 26.2.23, 0 < p <= 0.5
c Approximation to inverse of cdf standard normal
c Accurate approximation with one Newton Raphson iteration
c
c**********************************************************************
c
      FUNCTION INVNOR(p)

c     ..Parameters..
      double precision one
      parameter       (one=1.0d0)

c     ..Arguments..
      double precision INVNOR,p
c
c     ..Local..
      double precision a(3), b(4), t, a1, b1, to

c     ..Calling function..
      double precision POLYEV           !evaluation of polynomial
      double precision PDFNOR             !pdf of standard normal
      double precision CDFNOR             !cdf of standard normal

      DATA a/2.515517D0,0.802853D0,0.010328D0/
      DATA b/1.0D0,1.432788D0,0.189269D0,0.001308D0/

      t      = sqrt(log(one/(p*p)))
      a1     = POLYEV(3,a,t)
      b1     = POLYEV(4,b,t)
      to     = t - (a1/b1)

c..   One newton iteration to increase accuracy

      INVNOR = to - ((CDFNOR(to) + p - one)/PDFNOR(to))

      RETURN
      END
c
c*********************************************************************
c
c Abramowitz et al. 26.2.17
c Approximation to the cdf standard normal, x >= 0
c
c*********************************************************************
c
      FUNCTION CDFNOR(x)

c     ..Parameter..
      double precision one
      parameter       (one=1.0d0)

c     ..Arguments..
      double precision CDFNOR,x

c     ..Local..
      double precision t, a(6), pol

c     ..Calling functions..
      double precision POLYEV                !polynomial evaluation
      double precision PDFNOR                        !pdf of normal

      DATA a/1.0d0,0.319381530d0,-0.356563782d0,
     *       1.781477937d0,-1.821255978d0,1.330274429d0/

      t      = one/(one + 0.2316419d0*x)
      pol    = POLYEV(6,a,t) - one
      CDFNOR = one - (PDFNOR(x) * pol)

      RETURN
      END
c
c*********************************************************************
c
c     pdf of normal
c
c*********************************************************************
c
      FUNCTION PDFNOR(x)

c     ..Parameter..
      double precision  pi,two
      parameter        (pi=3.1415926535897931d0,two=2.0d0)

c     ..Arguments..
      double precision PDFNOR,x
  
      PDFNOR = exp( -x*x/two)/sqrt(two * pi)

      RETURN
      END
c
c*********************************************************************
c
c     To evaluate polynomial
c     P(x) = a_nx^n + ... a_1x + a_o at xo (Horner's method) 
c
c*********************************************************************
c
      FUNCTION POLYEV(n,coef,xo)
c
c     ..Arguments..
      integer          n,i
      double precision POLYEV,coef(n),xo,p

      p = coef(n)

      do 10 i = (n-1),1,-1
       p = p*xo + coef(i)
   10 continue

      POLYEV = p

      RETURN
      END
c
c*********************************************************************
c
