c***************************** CONTENTS **********************************
c
c        POSTER   : posterior distribution (m rows by n columns)
c        NORDENS  : Posterior matrix construction, given Normal
c
c*************************************************************************
c
c Posterior distribution of MPCP
c    pr(ij) = Pr{tau[i]=j:X,pn,nmu,R},i=1,m,j=1,n.
c **INPUT
c idum   = seed 
c count  = counter to inspect row cells
c jd     = Posterior distribution calculation from position jd+1
c X(m,n) = data matrix (Normal)
c nmu(m,2) = Matrix of two normal parameter means
c R(m,2)   = Matrix of two Precision parameters
c pn(n)    = multinomial variates (value changes on exit)
c tmpn,tmpm,itmpn = Dummy vectors
c **WORK
c Compute cell probabilities (pr), row marginals (pd) and cell/row (pr)
c Select change point column (tau)
c **OUTPUT
c tau(m) = Column selected with probability weight for row i=1,m
c
c*************************************************************************
c
      SUBROUTINE POSTER(idum,count,m,n,nvar,jd,
     *                  EPS,X,pn,nmu,R,tmpn,tmpm,itmpn,
     *                  pd,pr,tau)

c     ..Parameters..
      double precision  zero
      parameter        (zero=0.0d0)

c     ..Scalar arguments..
      integer          idum,count,m,n,jd
      double precision EPS

c     ..Array arguments..
      integer          tau(m),nvar(m),itmpn(n)
      double precision X(m,n),
     *                 pn(n),nmu(m,2),R(m,2),
     *                 tmpn(n),tmpm(m),
     *                 pd(m),pr(m,n)

c     ..Local scalars..
      integer          i,j,jds,nv
      double precision pds

c     ..Calling subroutine..
c     NORDENS                     !Normal densities for post dist
c     RMULTNM                          !column selection at row i

c     ..Compute Normal densities at each X(i,j) data matrix..
      call NORDENS(m,n,jd,count,EPS,tmpn,itmpn,
     *             X,nmu,R,nvar,pn,pr)

      jds = jd + 1

      do 20 i = 1,m                         !sum of row densities
       pds = zero
       nv  = nvar(i)
       do 10 j = jds,nv
        pds = pds + pr(i,j)
   10  continue
       pd(i) = pds
   20 continue

      do 40 i = 1,m                       !posterior distribution
       nv = nvar(i)
       do 30 j = jds,nv
        pr(i,j) = pr(i,j)/pd(i)
   30  continue
   40 continue

c     ..Select change point column position at each row..
c     ..at each row i, select a random sample of size 1 from
c     ..a set of distinct objects {1,2,...,n} with prob pr(i,)..

      do 60 i = 1,m
       nv = nvar(i)
       do 50 j = jds,nv
        pn(j) = pr(i,j)
   50  continue
       call RMULTNM(idum,1,nv,jd,pn,tmpn,tmpm,tau(i))
   60 continue

      RETURN
      END
c
c*************************************************************************
c
c  Matrix of posterior distribution construction 
c    given Normal distribution. We try to implement
c    `efficient' tricks suggested by Colemann and Van Loan (1988)
c
c*************************************************************************
c
      SUBROUTINE NORDENS(m,n,jd,count,EPS,tmpn,itmpn,
     *                   X,nmu,R,nvar,pn,pr)

c     ..Parameters..
      double precision pi,TOL,zero,two
      parameter       (pi=3.1415926535897931d0,TOL=707.0d0,
     *                 zero=0.0d0,two=2.0d0)       

c     ..Arguments..
      integer          m,n,jd,count,nvar(m),itmpn(n)
      double precision X(m,n),EPS,nmu(m,2),R(m,2),tmpn(n),
     *                 pn(n),pr(m,n)

c     ..Local..
      integer          i,j,k,jds,nv
      double precision stpi,mu1,mu2,R1,R2,lsR1,lsR2,
     *                 sumX1,sumsqX1,sumX2,sumsqX2,mintol

c     ..Calling procedures..
c     SHIFTP                         !shift values to avoid under-overflow

      jds = jd+1
      stpi = sqrt(two*pi)

      do 60 i = 1,m                                               !m rows
       mintol = -3.0d0*TOL
       nv     = nvar(i)
       mu1 = nmu(i,1)
       mu2 = nmu(i,2)
       R1 = R(i,1)
       R2 = R(i,2)
       lsR1 = log(sqrt(R1))
       lsR2 = log(sqrt(R2))

       do 40 j = jds,nv                                       !nv columns
         sumX1   = zero
         sumsqX1 = zero

         do 20 k = 1,j                                !row distribution 1
          sumX1   = sumX1   + X(i,k)
          sumsqX1 = sumsqX1 + X(i,k)*X(i,k)
   20    continue
         sumX1 = R1 * (sumsqX1 + mu1 * (j*mu1 - two*sumX1))

         if(j.lt.nv) then                             !row distribution 2
          sumX2   = zero
          sumsqX2 = zero
          do 30 k = (j+1),nv
           sumX2   = sumX2   + X(i,k)
           sumsqX2 = sumsqX2 + X(i,k)*X(i,k)
   30     continue
          sumX2 = R2 * (sumsqX2 + mu2 * ((nv-j)*mu2 - two*sumX2))
          sumX1 = sumX1 + sumX2
         endif

         sumX1  = -0.5d0*sumX1
         sumX1  = sumX1 - nv*log(stpi) +  j*lsR1 + (nv-j)*lsR2

c     ..Multinomial variates can be very, very small
        if (pn(j).gt.EPS) then
         sumX1  = sumX1 + log(pn(j))
        else                                                !pn(j) <= EPS
         sumX1  = sumX1 - (two*TOL)
        endif
        mintol  = MAX(sumX1,mintol)
        tmpn(j) = sumX1
   40  continue                                           !end nv columns
        
c     ..shift tmpn if abs(mintol) > TOL

       if (abs(mintol).gt.TOL)                             !shifting work
     *  call SHIFTP(TOL,i,count,nv,jd,tmpn,itmpn)

c     ..Compute product of row densities..

       do 50 j = jds,nv                                 !density elements
        pr(i,j) = exp(tmpn(j))
   50  continue

   60 continue                                            !end i=1,m rows

      RETURN
      END
c
c*************************************************************************
c
