c****************************CONTENTS********************************
c
c     READROWCOL     - Read number of rows/columns in parameter file
c     READAT         - Read data file 
c     READPAR        - Read parameter file
c     READRC         - Read selected rows or columns for updates
c     CHKINDEX       - Index of row/column < # row/column
c     READFIL        - Read file names
c     READMAT        - Read parameters in matrix form
c     DUMPAR         - Dump read parameters to file
c     NROWCOL        - Measure number of rows and columns to 
c                       output to file (max limrow rows/cols
c                       and/or reasonable size)
c     CHKITERS       - Number of iterations to output to file
c     CHKFILE        - Approximate measurement of file sizes
c     OUTPMAT        - Write parameters in matrix form
c     VARSAMP        - Variable row sample size vector
c     CHKNCH         - Pr(no change) output to file if space permits
c     MAXLIM         - Index of rows or columns to output updates
c     CHKPAR         - Initial parameter checks for Normal
c     INVECT         - Initialize parameter 'updates'
c     IDENT          - Identify missing value cells 
c
c********************************************************************
c
c     Read number of rows and columns contained in data matrix.
c     Check that rows,cols (>0) do not exceed limits rmax,cmax;
c      that tmax>=max(rows,cols) so that we define a working
c      vector to an appropriate length;
c      that tmax>=limrow so that index vectors are not too long
c
c********************************************************************
c
      SUBROUTINE READROWCOL(dfile,fname,rows,cols,rmax,cmax,
     1                      tmax,limrow)

c     ..Arguments..
      integer         rows,cols,rmax,cmax,tmax,limrow
      character*9     dfile,fname                 !file name and type

c     ..Local..
      logical         wrong

c     ..INITIALIZATION..
      wrong=.false.

      write(6,'(/2x,5a)')              'Open  ',fname,' file "',dfile,
     *                                                '" (READROWCOL)'
      open(7,FILE=dfile,STATUS='OLD',ERR=98)
      read(7,*,ERR=99) rows,cols
      close(7)
      write(6,'(2x,5a)')               'Close ',fname,' file "',dfile,
     *                                                '" (READROWCOL)'

c     ..Check empty cells..
      if ((rows.lt.1).or.(cols.lt.1)) then
       wrong=.true.
       write(6,'(/4x,2a/)')            'User specified number of rows',
     *                                    ' or columns < minimum of 1'
       GOTO 99
      endif

c     ..Check program allocation capacity..
      if ((rows.gt.rmax).or.(cols.gt.cmax)) then
       wrong=.true.
       write(6,'(/4x,2a,/6x,i7,a/)')   'User specified number of rows',
     *                 ' or columns exceed capacity of',max(rmax,cmax),
     *                                              ' rows or columns'
       GOTO 97
      endif

c     ..Check correct maximum vector length..
      if (tmax.lt.max(rows,cols)) then
       wrong=.true.
       write(6,'(/4x,2a/)')              'Program parameter TMAX is <',
     *                   ' maximum(number of rows, number of columns)'
       GOTO 97
      endif

c     ..To avoid garbage output, ensure that tmax>limrow..
      if (tmax.lt.limrow) then
       wrong=.true.
       write(6,'(/4x,2a/)')              'Program parameter TMAX is <',
     *                          ' number of selected rows/cols LIMROW'
       GOTO 97
      endif

   97 if (wrong) then
       write(6,'(/4x,2a)')      'Parameters to change in program MAIN',
     *                                    ' are one of the following:'
       write(6,'(6x,a)')              'rmax - maximum rows in program'
       write(6,'(6x,a)')           'cmax - maximum columns in program'
       write(6,'(6x,a)')   'tmax - program maximum(data rows,columns)'
       write(6,'(6x,a/)')'limrow - number of rows and columns to scan'
       STOP ' in READROWCOL (MPCP) - CORRECT and RECOMPILE'
      endif

      RETURN

c     ..Error traps..
   98 write(6,'(/4x,4a)')                   fname,' file: "',dfile,'"'
      write(6,'(6x,a/)')                               'is inexistent'
      STOP ' in READROWCOL (MPCP)'

   99 write(6,'(/4x,a)')                   'I/O error has occurred in'
      write(6,'(6x,4a/)')                   fname,' file: "',dfile,'"' 
      STOP ' in READROWCOL (MPCP)'

      END
c
c********************************************************************
c
c     Read data file
c     minel, maxel take minimum and maximum data values
c      valid for missing value or variable sample size indicator
c
c********************************************************************
c
      SUBROUTINE READAT(dfile,dumpf,fname,rows,cols,X,minel,maxel)

c     ..Parameters..
      double precision TOP,BOT
      parameter       (TOP=1.0d300,BOT=-1.0d300)

c     ..Arguments..
      integer          rows,cols
      double precision X(rows,cols),minel,maxel
      character*9      dfile,dumpf,fname         !file names and type

c     ..Local..
      integer           i,j

c     ..INITIALIZATION..
      minel = TOP
      maxel = BOT

      write(6,'(/2x,5a)')              'Open  ',fname,' file "',dfile,
     *                                                    '" (READAT)'
      open(7,FILE=dfile,STATUS='OLD',ERR=98)
      do 10 i = 1,rows                                !begin data read
        read(7,*,ERR=99) (X(i,j),j=1,cols)
   10  continue       
      close(7)                                          !end data read
      write(6,'(2x,5a)')               'Close ',fname,' file "',dfile,
     *                                                    '" (READAT)'

      do 20 i = 1,rows
       do 15 j = 1,cols
        minel = MIN(X(i,j),minel)
        maxel = MAX(X(i,j),maxel)
   15  continue
   20 continue
      write(6,'(/4x,a,g15.8)')        'minimum data element is ',minel
      write(6,'(4x,a,g15.8)')         'maximum data element is ',maxel

c      ..Dump data to file..

      open(8,FILE=dumpf)                           !write to dump file
      write(8,'(2a)')                                  fname,' read :'
      do 40 i = 1,rows
       write(8,'(10(1x,g12.5))') (X(i,j),j=1,cols)
   40 continue
      close(8)

      RETURN

c     ..Error traps..
   98 write(6,'(/4x,4a)')                   fname,' file: "',dfile,'"'
      write(6,'(6x,a/)')                               'is inexistent'
      STOP ' in READAT (DATAUGM)'

   99 write(6,'(/4x,a)')                   'I/O error has occurred in'
      write(6,'(6x,4a)')                    fname,' file: "',dfile,'"'
      write(6,'(5x,2(1x,a,1x,i5)/)')         'at row',i,'and column',j 
      STOP ' in READAT (DATAUGM)'

      END
c
c********************************************************************
c
c     Read parameter file
c
c********************************************************************
c
      SUBROUTINE READPAR(dfile,fname,m,n,mnlen,iseed,S,liter,mult,
     1                   multf,MISS,maxsize,misval,VAR,varval,jd,
     2                   limrow,agam,bgam,mu,omega,alpha,
     3                   samprow,sampcol,indx,
     4                   mpcalpn,mpcgam1,mpcgam2,mpcnor1,mpcnor2,
     5                   mpcavgd,mpcgexp,mpcndif,mpcpnch)

c     ..Arguments..
      integer          m,n,mnlen,iseed,S,liter,mult,multf,
     *                 jd,limrow,maxsize,
     *                 samprow(limrow),sampcol(limrow),indx(mnlen)
      double precision misval,varval,
     *                 agam(m,2),bgam(m,2),alpha(n),mu(m,2),omega(2)
      logical          MISS,VAR
      character*9      dfile,fname                 !file name and type
      character*15     mpcgam1,mpcgam2,mpcnor1,mpcnor2,mpcalpn,
     *                 mpcavgd,mpcgexp,mpcndif,mpcpnch   !output files

c     ..Local..
      character*10     mess1

c     ..Calling procedures..
c     READRC                                   !Read row/col selection
c     READFIL                                         !Read file names
c     READMAT                                  !Read matrix parameters

      write(6,'(/2x,5a)')              'Open  ',fname,' file "',dfile,
     *                                                   '" (READPAR)'
      open(7,FILE=dfile)                         !begin parameter read

c     ..Number of rows, columns; seed
      read(7,*,ERR=99) m,n,iseed

c     ..Number of updates, Last number to retain for output..
      read(7,*,ERR=99) S,liter
      if (liter.gt.S) then
       write(6,'(//4x,a)')      'Updates retained > Updates requested'
       write(6,'(4x,a/)')  'Set Updates retain ==== Updates requested'
       liter = S
      endif

c     ..screenview,filedumps,maxfilesize(kb)..
      read(7,*,ERR=99) mult,multf,maxsize 
      maxsize=maxsize*1000 

c     ..MISSing data (MISS=T/F,misval=integer (ignored in POISSON))
      read(7,*,ERR=99) MISS,misval 

c     ..VARiable sample size (VAR=T/F,varval=positive integer)
      read(7,*,ERR=99) VAR,varval
      write(6,'(4x,a)')                    'Variable sample size read'

c     ..Index for row selection (max=10 rows to select)
      mess1='rows' 
      call READRC(m,limrow,samprow,indx,mess1,fname,dfile)
c     ..Index for column selection (max=10 columns to select)
      mess1='cols'
      call READRC(n,limrow,sampcol,indx,mess1,fname,dfile)

c     ..File names to output iterated updates
      call READFIL(mpcgam1,mpcgam2,mpcnor1,mpcnor2,mpcalpn,
     *             mpcavgd,mpcgexp,mpcndif,mpcpnch) 

c     ..Gamma prior parameters (a1=shape,b1=scale),(a2=shape,b2=scale)
      mess1='Gamma'
      call READMAT(7,m,2,jd,agam,mess1,dfile,fname) 
      write(6,'(/4x,4(a,1x))')             'read',mess1,fname,'shapes'
      write(6,'(6x,1x,a,2(1x,e12.5))')               'Last item read:',
     *                                             agam(m,1),agam(m,2)
      call READMAT(7,m,2,jd,bgam,mess1,dfile,fname)
      write(6,'(/4x,4(a,1x))')             'read',mess1,fname,'scales'
      write(6,'(6x,1x,a,2(1x,e12.5))')               'Last item read:',
     *                                             bgam(m,1),bgam(m,2)

c     ..Normal prior parameters (mu=mean)
      mess1='Normal'
      call READMAT(7,m,2,jd,mu,mess1,dfile,fname)
      write(6,'(/4x,3(a,1x))')                      'read',mess1,fname
      write(6,'(6x,1x,a,2(1x,e12.5))')               'Last item read:',
     *                                                 mu(m,1),mu(m,2)

c     ..Precision prior parameters (Omega>0=precision)
      mess1='Omega'
      call READMAT(7,1,2,jd,omega,mess1,dfile,fname)
      write(6,'(/4x,3(a,1x))')                      'read',mess1,fname
      write(6,'(6x,1x,a,2(1x,e12.5))')               'Last item read:',
     *                                               omega(1),omega(2)

c     ..Dirichlet prior parameters 
      mess1='Dirichlet'
      call READMAT(7,1,n,jd,alpha,mess1,dfile,fname)
      write(6,'(/4x,3(a,1x))')                      'read',mess1,fname
      write(6,'(6x,1x,a,1(1x,e12.5))')               'Last item read:',
     *                                                        alpha(n)

      close(7)                                     !end parameter read
      write(6,'(2x,5a)')               'Close ',fname,' file "',dfile,
     *                                                   '" (READPAR)'

      RETURN

   99 write(6,'(/4x,a)')                   'I/O error has occurred in'
      write(6,'(6x,4a/)')                   fname,' file: "',dfile,'"'
      STOP ' in READPAR (DATAUGM)'

      END
c
c********************************************************************
c
c     Read selected rows or columns for updates.
c     Check that row/col indices do not exceed m,n of data matrix
c
c********************************************************************
c
      SUBROUTINE READRC(n,nobs,samp,indx,ltype,fname,dfile)

c     ..Arguments..
      integer        n,nobs,samp(nobs),indx(nobs)
      character*4    ltype
      character*9    fname,dfile

c     ..Local..
      integer        i,m
      logical        exceed

c     ..Calling procedures..
c     CHKINDEX

      m = min(n,nobs)
      exceed=.false.

      write(6,'(/)')
      if(n.gt.nobs) then
       write(6,'(4x,a,i4,a5)')   'Data matrix contains more than',
     *                                                nobs,ltype
       write(6,'(6x,a,i5,a)') 'Updates for only',nobs,' selected'
       write(6,'(6x,a4,a)')          ltype,' are output to files'
      endif

      read(7,*,ERR=99) (samp(i),i=1,m)
      write(6,'(4x,a,a4,a)')           'Selection ',ltype,' read'

      call CHKINDEX(n,m,samp,indx,exceed)
      if (exceed) then
       write(6,'(/4x,a,a5,a)')      'At least one',ltype,' index'
       write(6,'(6x,a,a5,a)')         'exceeds number of',ltype,
     *                                          ' in data matrix'
       write(6,'(10x,a)')                            '-   OR   -'
       write(6,'(6x,a/)')         'at least one index value zero'
       STOP ' in READRC (READPAR)'
      endif

      RETURN

c     ..Error traps..
   99 write(6,'(/4x,a,a4,a)')       'Index of ',ltype,' error in'
      write(6,'(6x,4a/)')              fname,' file: "',dfile,'"'
      STOP  ' in READRC (READPAR)'

      END
c
c********************************************************************
c
c     Check row/column index x < n of data matrix
c     x is vector of size limit > 1. Obviously, indices > n
c      or < 0 are rejected
c
c********************************************************************
c
      SUBROUTINE CHKINDEX(n,limit,x,indx,excess)

c     ..Arguments..
      integer          n,limit,x(limit),indx(limit)
      logical          excess

c     ..Calling procedure..
c     INDEXI                           !index sort integer array

      if (limit.gt.1) then 
       call INDEXI(limit,x,indx)
      else
       indx(limit) = 1
      endif

      if ((x(indx(limit)).gt.n).or.(x(indx(1)).lt.1)) excess=.true. 

      RETURN
      END
c
c********************************************************************
c
c     Read File names
c
c********************************************************************
c
      SUBROUTINE READFIL(mpcgam1,mpcgam2,mpcnor1,mpcnor2,mpcalpn,
     1                   mpcavgd,mpcgexp,mpcndif,mpcpnch)

c     ..Arguments..
      character*15     mpcgam1,mpcgam2,mpcalpn,
     *                 mpcnor1,mpcnor2,mpcavgd,mpcgexp,mpcndif,mpcpnch

c     ..Local..
      integer          count

      count = 0
      read(7,'(a)') mpcgam1                !m gamma(a1,b1) parameters
      count = count + 1
      read(7,'(a)') mpcgam2                !m gamma(a2,b2) parameters
      count = count + 1
      read(7,'(a)') mpcnor1                              !m N(mu1,R1)
      count = count + 1
      read(7,'(a)') mpcnor2                              !m N(mu2,R2)
      count = count + 1
      read(7,'(a)') mpcalpn                                 !n alphas
      count = count + 1
      read(7,'(a)') mpcavgd           !n avg (iterations) dir updates
      count = count + 1
      read(7,'(a)') mpcgexp          !mx2 E(Xi),Xi~Gamma(ai,bi),i=1,2
      count = count + 1
      read(7,'(a)') mpcndif     !m E(X2)-E(X1),Xi~Normal(ui,si),i=1,2
      count = count + 1
      read(7,'(a)') mpcpnch             !net iterations Pr(no change)
      count = count + 1

      write(6,'(/4x,a,i2,a)')            'read ',count,' file names:'
      write(6,'(6x,2a)')   'Distribution 1 gamma updates:  ',mpcgam1
      write(6,'(6x,2a)')   'Distribution 2 gamma updates:  ',mpcgam2 
      write(6,'(6x,2a)')   'Distribution 1 normal updates: ',mpcnor1 
      write(6,'(6x,2a)')   'Distribution 2 normal updates: ',mpcnor2
      write(6,'(6x,2a)')   'Dirichlet updates:             ',mpcalpn 
      write(6,'(6x,2a)')   'Dirichlet updates averaged:    ',mpcavgd 
      write(6,'(6x,2a)')   'Gamma updates averaged:        ',mpcgexp
      write(6,'(6x,2a)')   'Avg diff mean updates Normals: ',mpcndif
      write(6,'(6x,2a)')   'Probability of no change:      ',mpcpnch

      RETURN
      END
c
c********************************************************************
c
c     Read parameters in matrix form
c     jd is for Dirichlet, where if jd=0, no Dirichlet
c      update is computed.
c
c********************************************************************
c
      SUBROUTINE READMAT(u1,m,n,jd,x,messg,dfile,fname)

c     ..Parameters..
      double precision zero
      parameter       (zero=0.0d0)

c     ..Arguments..
      integer          u1,m,n,jd
      double precision x(m,n)
      character*9      messg,
     *                 dfile,fname               !file name and type

c     ..Local..
      integer          i,j,k

c     ..Read Prior Parameters..
      if (messg.eq.'Normal') then
       do 10 j = 1,n
         read(u1,*,ERR=99) (x(i,j),i=1,m)
   10  continue
      else
       do 20 i=1,m 
        read(u1,*,ERR=99) (x(i,j),j=1,n)
   20  continue
      endif
     
c     ..Checks on valid parameter values..
      if ((messg.eq.'Gamma').or.(messg.eq.'Omega')) then
       do 40 i = 1,m
        do 30 j = 1,n
         if (x(i,j).le.zero) then
          write(6,'(4x,a,1x,i5)')      'Illegal value (<=0) at row',i
          write(6,'(6x,a,1x,i5,1x,3a/)')            'column',j,'of ',
     *                                             messg,' parameter'
          STOP ' in READMAT (READPAR)'
         end if
   30   continue
   40  continue
      endif

c     ..Set Dirichlet parameters to zero up to last zero..
      if (messg.eq.'Dirichlet') then
       jd = 0                                !last Dirichlet parm <=0
       i  = 1
       do 50 j = 1,n
        if (x(i,j).le.zero) jd=j
   50  continue
       do 60 j = 1,jd,1                !set first j dirichlet to zero
        x(i,j) = zero
   60  continue
      endif

c     ..Check for EOF (if TRUE=GOTO 999) (NO GUARANTY)..
      if (messg.eq.'Dirichlet') then 
       read(u1,*,ERR=999) k 
       write(6,'(/4x,a)')           'I/0 error: There is more data in'
       write(6,'(6x,4a/)')                  fname,' file: "',dfile,'"'
       STOP ' in READMAT (READPAR)'
      endif

  999 RETURN

   99 write(6,'(/4x,a)')                   'I/O error has occurred in'
      write(6,'(6x,4a/)')                   fname,' file: "',dfile,'"'
      STOP ' in READMAT (READPAR)'

      END
c
c********************************************************************
c
c     Check and dump parameter information
c
c********************************************************************
c
      SUBROUTINE DUMPAR(dumpf,fname,m,n,X,iseed,S,liter,mult,multf,
     1                  MISS,misval,VAR,varval,minel,maxel,jd,
     2                  limrow,nrow,ncol,agam,bgam,mu,omega,alpha,
     3                  samprow,sampcol,nvar,
     4                  filecnt,countf,maxsize,opendirf,opengamf,
     5                  mpcavgd,mpcgexp,mpcpnch)

c     ..Arguments..
      integer          m,n,iseed,S,liter,mult,multf,
     *                 jd,limrow,nrow,ncol,
     *                 filecnt,countf,maxsize,
     *                 samprow(limrow),sampcol(limrow),nvar(m)
      double precision misval,varval,minel,maxel,X(m,n),
     *                 agam(m,2),bgam(m,2),
     *                 alpha(n),mu(m,2),omega(2)
      logical          MISS,VAR,opendirf,opengamf
      character*9      dumpf,fname                !file name and type
      character*15     mpcavgd,mpcgexp,mpcpnch

c     ..Local..
      integer          i
      character*10     mess1,mess2,mess3

c     ..Calling procedures..
c     NROWCOL                   !Number of rows/cols to output to file
c     CHKITERS                 !Number of iterations to output to file
c     CHKFILE                   !Ensure that file size is not too huge
c     OUTPMAT                         !Output to file parameter inputs
c     VARSAMP                                !Variable row sample size
c     CHKNCH                         !Pr(no change) check with VARSAMP
c     MAXLIM                   !Index of row/column for update outputs
c     CHKPAR                  !No confusion with samp size and missing

      call NROWCOL(limrow,m,n,nrow,ncol)
      call CHKITERS(S,liter,multf,filecnt,countf)
      mess1='Dirichlet'
      call CHKFILE(opendirf,ncol,countf,
     *             maxsize,mpcavgd,mess1)
      mess1='Gamma'
      call CHKFILE(opengamf,nrow,countf,
     *             maxsize,mpcgexp,mess1)
      mess1='Normal'
      write(6,'(//4x,a10,a)') mess1,' updates the same size.'

      write(6,'(/2x,5a)')              'Open  ',fname,' file "',dumpf,
     *                                                    '" (DUMPAR)'
      open(8,FILE=dumpf)

      write(8,'(2a)')                                  fname,' read :'
      write(8,'(a)')                      'Number of rows and columns'
      write(8,'(2(i15,1x))') m,n
      write(8,'(1x,a,i10)')      'Seed for random number generators: ',
     *                                                           iseed

      if (iseed.gt.0) iseed=-iseed     !To conform to number generator

      write(8,'(1x,a)')             'Number of iterations are set at:'
      write(8,'(3x,a,i15)')            'Total number of iterations:',S
      write(8,'(3x,a,i15)')      'Last iterations to output to file: ',
     *                                                           liter
      write(8,'(2(1x,a),i15,a)')        'From last iterations, output',
     *                               ' to file every ',multf,'th time' 
      write(8,'(1x,a,i15,a)') 'Output to screen every ',mult,'th time'
      write(8,'(1x,a,i15)')            'Maximum file size is ',maxsize
      write(8,'(a)')                     'Missing (logical) and value'
      write(8,'(L5,1x,g12.5)') MISS,misval
      write(8,'(a)')           'Sample size for each row indicated by'
      write(8,'(2x,a)')        'Variable size (logical) and indicator'
      write(8,'(L5,1x,g12.5)') VAR,varval

      mess1='Gamma'
      mess2='shapes'
      mess3='rows'
      call OUTPMAT(8,m,2,agam,mess1,mess2,mess3)

      mess2='scales'
      call OUTPMAT(8,m,2,bgam,mess1,mess2,mess3)

      mess1='Normal'
      mess2='means'
      call OUTPMAT(8,m,2,mu,mess1,mess2,mess3)

      mess1='Omega'
      mess2='parameters'
      call OUTPMAT(8,1,2,omega,mess1,mess2,mess3)

      mess1='Dirichlet'
      mess2='parameters'
      mess3='columns'
      call OUTPMAT(8,1,n,alpha,mess1,mess2,mess3)

      write(8,'(a,i5,1x,2a)')      'First ',jd,'Dirichlet parameters',
     *                                             ' are set to zero'

c     ..Variable row sample size
      call VARSAMP(m,n,VAR,varval,X,nvar)

c     ..Pr(no change) check (in terms of nvar)..
      call CHKNCH(m,n,nvar,opendirf,mpcpnch)

      write(8,'(a,1x,i5,1x,a)')             'Sample size for each of',
     *                                                   m,'rows is:'
      write(8,'(10(1x,i5))') (nvar(i),i=1,m)

c     ..Index of Maximum number of rows / columns for update outputs
      mess1='Gam/Norm'
      mess2='row updates'
      call MAXLIM(limrow,nrow,samprow,opengamf,
     *            8,mess1,mess2)

      mess1='Dirichlet'
      mess2='col updates'
      call MAXLIM(limrow,ncol,sampcol,opendirf,
     *            8,mess1,mess2)

      close(8)
      write(6,'(/2x,5a)')             'Close ',fname,' file "',dumpf,
     *                                                   '" (DUMPAR)'

c     ..Check missing and variable parameter values..
      call CHKPAR(MISS,VAR,misval,varval,minel,maxel,
     *            jd,m,nvar)

      RETURN
      END
c
c********************************************************************
c
c     Number of columns and rows to retain for update output
c
c********************************************************************
c
      SUBROUTINE NROWCOL(limit,m,n,nrow,ncol)

c     ..Arguments
      integer         limit,m,n,nrow,ncol

      if (m.gt.limit) then
       nrow = limit
      else
       nrow = m
      endif

      if (n.gt.limit) then
       ncol = limit
      else
       ncol = n
      endif

      RETURN
      END
c
c********************************************************************
c
c     Number of iterations to output to file
c     filecnt is number of iterations retained regardless of 
c      the size of multf; countf is net number of iterations
c      retained; 
c
c********************************************************************
c
      SUBROUTINE CHKITERS(totaliter,retainiter,multf,filecnt,countf)

c     ..Arguments..
      integer      totaliter,retainiter,multf,filecnt,countf

      if ((totaliter.le.0).or.(retainiter.le.0)) then
       write(6,'(/4x,2a/)') 'Please set number of iterations',
     *    ' (retained or otherwise) > 0'
       STOP ' in CHKITERS (DUMPAR)'
      endif

      if (multf.le.1) then
       multf   = 1
       filecnt = totaliter - retainiter + multf
       countf  = retainiter
      elseif (multf.ge.retainiter) then
       multf   = retainiter
       filecnt = totaliter
       countf  = 1
      elseif (mod(retainiter,multf).eq.0) then
       filecnt = totaliter - retainiter + multf
       countf  = retainiter/multf
      else
       filecnt = totaliter - retainiter +
     *            mod(retainiter,multf)
       countf  = nint(dble(retainiter)/dble(multf) + 0.50d0)
      endif

      RETURN
      END
c
c********************************************************************
c
c     Measure size of files in bytes (effective size 'cursize' 
c      compared to maximum size 'maxsize') in order to decide 
c      whether or not file is opened or if only summary file
c      is opened.
c     Parameter colns is set to 3 because column 1 is for
c      parameter 1, column 2 is for parameter 2 and column3
c      is for an indicator to denote whether row has had
c      a change in distribution before the last column
c
c********************************************************************
c
      SUBROUTINE CHKFILE(openfile,nrc,countf,
     1                   maxsize,favg,mesg)

c     ..Parameters..
      integer         bytesize,colns
      parameter      (bytesize=13,colns=3)

c     ..Arguments..
      integer         nrc,countf,maxsize
      logical         openfile
      character*15    favg
      character*10    mesg

c     ..local..
      integer         cursize,cursize1

      cursize1 = countf*bytesize
      cursize  = cursize1*nrc

      if (mesg.ne.'Dirichlet') then
       cursize  = cursize *colns
       cursize1 = cursize1*colns
      endif

      openfile = .true.
      if ((cursize).lt.maxsize) then
       write(6,'(//4x,a10,a)')        mesg,' updates will be output'
       write(6,'(6x,a,i5,a)')                       'over the ',nrc,
     *                                    ' first row/col selected.'
      elseif ((cursize1).lt.maxsize) then
       write(6,'(//4x,a10,a)')        mesg,' updates will be output'
       write(6,'(6x,a)')     'only over the first row/col selected.'
       write(6,'(4x,a,i5)')     'Approximate size for desired ',nrc
       write(6,'(6x,a,i12)')            'rows/columns was ',cursize
       write(6,'(6x,2a,i12)')               'which exceeds maximum',
     *                                  ' allowed size of ',maxsize  
       nrc = 1
      else
       openfile = .false.
       write(6,'(//4x,a10,a)')    mesg,' updates will not be output'
       write(6,'(6x,a)')       'over the selected row/col. User can'
       write(6,'(6x,a,a15)')      'inspect only summary file ',favg
       write(6,'(4x,a)')       'Approximate size for desired single'
       write(6,'(6x,a,i12)')             'row/column was ',cursize1
       write(6,'(6x,2a,i12)')               'which exceeds maximum',
     *                                  ' allowed size of ',maxsize  
      endif

      RETURN
      END
c
c********************************************************************
c
c     Output parameter information in matrix form
c
c********************************************************************
c
      SUBROUTINE OUTPMAT(u1,m,n,x,mess1,mess2,mess3)

c     ..Arguments..
      integer           u1,m,n
      double precision  x(m,n)
      character*10      mess1,mess2,mess3

c     ..Local..
      integer          i,j,size

      size=n
      if (mess3.eq.'rows') size=m
       write(8,'(2a,i4,1x,a)') mess1,mess2,size,mess3

       do 10 i = 1,m
         write(u1,'(5(1x,g12.5))') (x(i,j),j=1,n)
   10  continue
      
      RETURN
      END
c
c********************************************************************
c
c     Variable row sample size in data matrix X;
c     If sample size of each row is variable, then
c      the calculation of Probability of No change is
c      meaningless (see CHKNCH).
c
c********************************************************************
c
      SUBROUTINE VARSAMP(m,n,VAR,varval,X,nvar)

c    ..Arguments..
      integer          m,n,nvar(m)
      double precision varval,X(m,n)
      logical          VAR

c    ..Local..
      integer          i,j

      do 20 i = 1,m                         !default sample size
       nvar(i) = n
   20 continue

      if (VAR) then                    !is sample size variable?
       do 40 i = 1,m
        do 30 j = 1,n
         if (X(i,j).ge.varval) then 
          nvar(i) = j-1                    !sample size of row i
          GOTO 40
         endif
   30   continue
   40  continue
      endif

      RETURN
      END
c
c********************************************************************
c
c    Output probability of no change if space permits. 
c    Warning when row samples are of variable size (see VARSAMP)
c
c********************************************************************
c
      SUBROUTINE CHKNCH(m,n,nv,allow,fprnch)
c
c     ..Arguments..
      integer      m,n,nv(m)                           !nv(i),i=1,n
      logical      allow                         !can we open file?
      character*15 fprnch              !file name for Pr(no change)

c     ..Local..
      integer      i
      logical      varies

      if (.not.allow) then
       write(6,'(//4x,a)')        'Pr(no change) will not be output'
       write(6,'(6x,a,a15)')                      'to file ',fprnch
       write(6,'(6x,a)') 'because of insufficient file size request'
       RETURN
      endif

      varies = .false.
      do 100 i = 1,m
       if (nv(i).lt.n) then                   !sample size at row i
        varies = .true.
        GOTO 200
       endif
  100 continue

  200 if (varies) then
       write(6,'(//4x,a)')         'At least one row of data matrix'
       write(6,'(6x,a,i7)')  'has length less than fixed column ',n
       write(6,'(6x,a)')        'Pr(no change) is thus meaningless.'
       write(6,'(4x,a)')       'SUGGESTION: use missing data option'
       write(6,'(6x,a)')      'to generate variates in data matrix.'
      endif

      RETURN
      END
c
c********************************************************************
c
c     Index of rows and columns to output to dumpfile (u1) 
c
c********************************************************************
c
      SUBROUTINE MAXLIM(nrc,len,sampvec,allow,
     1                  u1,mess1,mess2)

c     ..Arguments..
      integer          nrc,len,sampvec(len),u1
      logical          allow
      character*10     mess1,mess2

c     ..Local..
      integer          j

      write(u1,'(a,1x,i4,3(1x,a))')               'Maximum of',nrc,
     *                           mess2,'indices selected for',mess1
      if (allow) then
       write(u1,'(10i5)') (sampvec(j),j=1,len)        !to dump file
      else
       write(8,'(1x,2a)')             '.....No output for ', mess2
      endif

      RETURN
      END
c
c********************************************************************
c
c     Some little checks: 
c     jd is the number of first dirichlet parameters set to zero
c     nvar(i) is the row sample size, i=1,m. nvar > jd
c     Row variable sample size indicator is BIG == maximum data 
c     Specific to Normal distribution: 
c     Missing value indicator is -BIG and == minimum data
c  
c********************************************************************
c
      SUBROUTINE CHKPAR(MISS,VAR,misval,varval,minel,maxel,
     1                  jd,row,nvar)

c     ..Arguments..
      double precision zero
      parameter       (zero=0.0d0)

c     ..Arguments..
      integer          jd,row,nvar(row)
      double precision misval,varval,minel,maxel
      logical          MISS,VAR

c     ..Local..
      integer          i
      logical          bad

c     ..INITIALIZATION..
      bad = .false.
      write(6,'(/4x,a)')                    'BEGIN parameter checks' 

      if (MISS.and.(misval.ge.zero)) then
       write(6,'(/6x,a/,8x,a/)') 'By convention, please set missing',
     *                      'value indicator to BIG NEGATIVE number'
       bad = .true.
       GOTO 100
      endif


      if (MISS.and.(misval.gt.minel)) then
       write(6,'(/6x,2a/,8x,a,1x,g17.9/)')   'By convention, please',
     *                                          ' set missing value',
     *                                  ' indicator equal to',minel
       bad = .true.
       GOTO 100
      endif

      if (VAR.and.(varval.le.zero)) then
       write(6,'(/6x,a/,8x,a/)')'By convention, please set variable',
     *          'sample size value indicator to BIG positive number'
       bad = .true.
       GOTO 100
      endif

      if (VAR.and.(varval.lt.maxel)) then
       write(6,'(/6x,2a/,8x,a,1x,g17.9/)')   'By convention, please',
     *                                   ' set variable sample size',
     *                                  ' indicator equal to',maxel
       bad = .true.
       GOTO 100
      endif

      do 50 i = 1,row
       if ((jd.ge.nvar(i)).or.(nvar(i).eq.0)) then
        write(6,'(/6x,i4,2a,i4/,8x,2a/)')     i,'th row sample size',
     *                          ' below minimum requirement of',jd+1,
     *                             'Check dirichlet parameter setup',
     *                               ' or row sample size indicator'
        bad = .true.
        GOTO 100
       endif
   50 continue

  100 if (bad) then
       STOP ' in CHKPAR (DUMPAR)'
      endif
      write(6,'(/4x,a)')                     'END   parameter checks'

      RETURN
      END
c
c********************************************************************
c
c    Initialize parameter 'updates' 
c
c********************************************************************
c
      SUBROUTINE INVECT(m,n,x,newx)
c
c     ..Arguments..
      integer           m,n
      double precision  x(m,n),newx(m,n)

c     ..Local..
      integer           i,j

      do 20 i = 1,m
       do 10 j = 1,n
        newx(i,j) = x(i,j)
   10  continue
   20 continue

      RETURN
      END
c
c********************************************************************
c
c     Get x,y coordinates of missing values
c
c********************************************************************
c
      SUBROUTINE IDENT(dumpf,fname,m,n,msamp,misval,X,cord)

c     ..Arguments..
      integer          m,n,msamp(m),cord(m,n)
      double precision misval,X(m,n)
      character*9      dumpf,fname                   !file type name  

c     ..Local..
      integer          i,j

      write(6,'(/2x,5a)')             'Open  ',fname,' file "',dumpf,
     *                                                    '" (IDENT)'
      open(8,FILE=dumpf)

      write(8,'(/2a)')                                fname,' read :'

      do 100 i = 1,m
       do 50 j = 1,msamp(i)
        cord(i,j) = 0
        if (X(i,j).le.misval) cord(i,j) = 1
   50  continue
        write(8,'(20i2)') (cord(i,j),j=1,msamp(i))
  100 continue

      close(8)
      write(6,'(/2x,5a)')             'Close ',fname,' file "',dumpf,
     *                                                    '" (IDENT)'

      RETURN
      END
c
c********************************************************************
c
