c***************************** CONTENTS **********************************
c
c        POSTER   : posterior distribution (m rows by n columns)
c        POIDENS  : Posterior matrix construction, given Poisson
c
c*************************************************************************
c
c Posterior distribution of MPCP
c    pr(ij) = Pr{tau[i]=j:X,lambda,pn},i=1,m,j=1,n.
c **INPUT
c idum   = seed 
c count  = counter to inspect row cells
c jd     = Posterior distribution calculation from position jd+1
c X(m,n) = data matrix (Poisson)
c lambda(m,2) = matrix of two Poisson parameters
c
c pn(n)    = multinomial variates (value changes on exit)
c tmpn,tmpm,itmpn = Dummy vectors
c **WORK
c Compute cell probabilities (pr), row marginals (pd) and cell/row (pr)
c Select change point column (tau)
c **OUTPUT
c tau(m) = Column selected with probability weight for row i=1,m
c
c*************************************************************************
c
      SUBROUTINE POSTER(idum,count,m,n,nvar,jd,
     *                  EPS,X,pn,lambda,tmpn,tmpm,itmpn,
     *                  pd,pr,tau)

c     ..Parameters..
      double precision  zero
      parameter        (zero=0.0d0)
              
c     ..Scalar arguments..
      integer          idum,count,m,n,jd
      double precision EPS

c     ..Array arguments..
      integer          tau(m),nvar(m),itmpn(n),
     *                 X(m,n)
      double precision pn(n),lambda(m,2),
     *                 tmpn(n),tmpm(m),
     *                 pd(m),pr(m,n)

c     ..Local scalars..
      integer          i,j,jds,nv
      double precision pds

c     ..Calling subroutine..
c     POIDENS                    !Poisson densities for post dist
c     RMULTNM                          !column selection at row i

c     ..Compute Poisson densities at each X(i,j) data matrix..
      call POIDENS(m,n,jd,count,EPS,tmpn,itmpn,
     *             X,lambda,nvar,pn,pr)

      jds = jd + 1

      do 20 i = 1,m                         !sum of row densities
       pds = zero
       nv  = nvar(i)
       do 10 j = jds,nv
        pds = pds + pr(i,j)
   10  continue
       pd(i) = pds
   20 continue

      do 40 i = 1,m                       !posterior distribution
       nv = nvar(i)
       do 30 j = jds,nv
        pr(i,j) = pr(i,j)/pd(i)
   30  continue
   40 continue

c     ..Select change point column position at each row..
c     ..at each row i, select a random sample of size 1 from
c     ..a set of distinct objects {1,2,...,n} with prob pr(i,)..

      do 60 i = 1,m
       nv = nvar(i)
       do 50 j = jds,nv
        pn(j) = pr(i,j)
   50  continue
       call RMULTNM(idum,1,nv,jd,pn,tmpn,tmpm,tau(i))
   60 continue

      RETURN
      END
c
c*************************************************************************
c
c  Matrix of posterior distribution construction 
c    given Poisson distribution. We try to implement
c    `efficient' tricks suggested by Colemann and Van Loan (1988)
c
c*************************************************************************
c
      SUBROUTINE POIDENS(m,n,jd,count,EPS,tmpn,itmpn,
     *                   X,lambda,nvar,pn,pr)

c     ..Parameters..
      double precision TOL,zero,two
      parameter       (TOL=707.0d0,zero=0.0d0,two=2.0d0)


c     ..Arguments..
      integer          m,n,jd,count,nvar(n),itmpn(n),X(m,n)
      double precision EPS,lambda(m,2),tmpn(n),
     *                 pn(n),pr(m,n)

c     ..Local..
      integer          i,j,k,jds,nv
      double precision sumX1,sumX2,mintol,la1,la2,lla1,lla2


c     ..Calling procedures..
c     SHIFTP                         !shift values to avoid under-overflow

      jds = jd+1


      do 60 i = 1,m                                               !m rows
       mintol = -3.0d0*TOL
       nv     = nvar(i)
       la1    = lambda(i,1)
       la2    = lambda(i,2)
       lla1   = log(la1)
       lla2   = log(la2)



       do 40 j = jds,nv                                       !nv columns
         sumX1   = zero


         do 20 k = 1,j                                !row distribution 1
          sumX1   = sumX1 + X(i,k)
   20    continue
         sumX1 = sumX1*lla1


         if(j.lt.nv) then                             !row distribution 2
          sumX2   = zero
          do 30 k = (j+1),nv
           sumX2   = sumX2   + X(i,k)
   30     continue
          sumX2 = sumX2*lla2
          sumX1 = sumX1 + sumX2
         endif



         sumX1 = sumX1 - j*la1 - (nv-j)*la2


c     ..Multinomial variates can be very, very small
        if (pn(j).gt.EPS) then
         sumX1  = sumX1 + log(pn(j))
        else                                                !pn(j) <= EPS
         sumX1  = sumX1 - (two*TOL)
        endif
        mintol  = MAX(sumX1,mintol)
        tmpn(j) = sumX1
   40  continue                                           !end nv columns
    
c     ..shift tmpn if abs(mintol) > TOL

       if (abs(mintol).gt.TOL)                             !shifting work
     *  call SHIFTP(TOL,i,count,nv,jd,tmpn,itmpn)

c     ..Compute product of row densities..

       do 50 j = jds,nv                                 !density elements
        pr(i,j) = exp(tmpn(j))
   50  continue

   60 continue                                            !end i=1,m rows

      RETURN
      END
c
c*************************************************************************
c
